"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseStep = parseStep;
exports.parseStepRef = parseStepRef;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const bitrise_1 = require("../../datasource/bitrise");
const git_tags_1 = require("../../datasource/git-tags");
function parseStep(stepRef, defaultRegistry) {
    if (is_1.default.emptyString(stepRef)) {
        return null;
    }
    const dep = {
        datasource: bitrise_1.BitriseDatasource.id,
        replaceString: stepRef,
    };
    const [ref, currentValue] = stepRef.split('@', 2);
    const refDep = parseStepRef(ref, defaultRegistry);
    // no version
    if (is_1.default.nullOrUndefined(currentValue)) {
        return {
            ...dep,
            packageName: stepRef,
            skipReason: 'unspecified-version',
        };
    }
    return {
        ...dep,
        ...refDep,
        currentValue,
    };
}
function parseStepRef(ref, defaultRegistry) {
    // handle local path
    // https://devcenter.bitrise.io/en/references/steps-reference/step-reference-id-format.html
    if (ref.startsWith('path::')) {
        return {
            depName: ref.split('::', 2)[1],
            skipReason: 'local-dependency',
        };
    }
    // handle Git references
    // https://devcenter.bitrise.io/en/references/steps-reference/step-reference-id-format.html
    if (ref.startsWith('git::')) {
        const [, packageName] = ref.split('::');
        return {
            packageName,
            datasource: git_tags_1.GitTagsDatasource.id,
        };
    }
    // step library references
    // https://devcenter.bitrise.io/en/references/steps-reference/step-reference-id-format.html
    const splitted = ref.split('::', 2);
    // reference which uses default registry
    // - script:
    if (splitted.length === 1) {
        const [packageName] = splitted;
        return {
            packageName,
            datasource: bitrise_1.BitriseDatasource.id,
            registryUrls: defaultRegistry ? [defaultRegistry] : undefined,
        };
    }
    // reference which overwrites Bitrise registry
    // https://github.com/bitrise-io/bitrise-steplib.git::script@1:
    const [registryUrl, packageName] = splitted;
    return {
        packageName,
        datasource: bitrise_1.BitriseDatasource.id,
        registryUrls: [registryUrl],
    };
}
//# sourceMappingURL=utils.js.map