"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const util_1 = require("./util");
function extractPackageFile(content, packageFile, config) {
    const deps = [];
    try {
        const lines = content
            .replaceAll(/^\s*\r?\n/gm, '') // replace empty lines
            .replaceAll(/^\s*#.*\r?\n/gm, '') // replace comment lines
            .split(regex_1.newlineRegex);
        const len = lines.length;
        for (let lineIdx = 0; lineIdx < len; lineIdx++) {
            const line = lines[lineIdx];
            const dockerImageObjectGroups = util_1.dockerImageObjectRegex.exec(line)?.groups;
            if (dockerImageObjectGroups) {
                // image object
                // https://support.atlassian.com/bitbucket-cloud/docs/docker-image-options/
                lineIdx = (0, util_1.addDepFromObject)(deps, lines, lineIdx, len, dockerImageObjectGroups.spaces, config.registryAliases);
                continue;
            }
            const pipeMatch = util_1.pipeRegex.exec(line);
            if (pipeMatch) {
                const pipe = pipeMatch[1];
                if (pipe.startsWith('docker://')) {
                    const currentPipe = pipe.replace('docker://', '');
                    (0, util_1.addDepAsDockerImage)(deps, currentPipe, config.registryAliases);
                }
                else {
                    (0, util_1.addDepAsBitbucketTag)(deps, pipe);
                }
                continue;
            }
            const dockerImageMatch = util_1.dockerImageRegex.exec(line);
            if (dockerImageMatch) {
                const currentFrom = dockerImageMatch[1];
                (0, util_1.addDepAsDockerImage)(deps, currentFrom, config.registryAliases);
            }
        }
    }
    catch (err) /* istanbul ignore next */ {
        logger_1.logger.debug({ err, packageFile }, 'Error extracting Bitbucket Pipes dependencies');
    }
    if (!deps.length) {
        return null;
    }
    return { deps };
}
//# sourceMappingURL=extract.js.map