"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateArtifacts = updateArtifacts;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const packageCache = tslib_1.__importStar(require("../../../util/cache/package"));
const hash_1 = require("../../../util/hash");
const http_1 = require("../../../util/http");
const promises_1 = require("../../../util/promises");
const regex_1 = require("../../../util/regex");
const common_1 = require("./common");
const http = new http_1.Http('bazel');
function getUrlFragments(rule) {
    const urls = [];
    const urlRecord = rule.children.url;
    if (urlRecord?.type === 'string') {
        urls.push(urlRecord);
    }
    const urlsRecord = rule.children.urls;
    if (urlsRecord?.type === 'array') {
        for (const urlRecord of urlsRecord.children) {
            if (urlRecord.type === 'string') {
                urls.push(urlRecord);
            }
        }
    }
    return urls;
}
const urlMassages = {
    'bazel-skylib.': 'bazel_skylib-',
    '/bazel-gazelle/releases/download/0': '/bazel-gazelle/releases/download/v0',
    '/bazel-gazelle-0': '/bazel-gazelle-v0',
    '/rules_go/releases/download/0': '/rules_go/releases/download/v0',
    '/rules_go-0': '/rules_go-v0',
};
function massageUrl(url) {
    let result = url;
    for (const [from, to] of Object.entries(urlMassages)) {
        result = result.replace(from, to);
    }
    return result;
}
function migrateUrl(url, upgrade) {
    const newValue = upgrade.newValue?.replace((0, regex_1.regEx)(/^v/), '');
    // @see https://github.com/bazelbuild/rules_webtesting/releases/tag/0.3.5
    // @see https://github.com/bazelbuild/rules_webtesting/releases/tag/0.4.0
    if (url.endsWith('/rules_webtesting.tar.gz') &&
        !newValue?.match((0, regex_1.regEx)(/^0\.[0123]\./))) {
        return url.replace((0, regex_1.regEx)(/\.tar\.gz$/), `-${newValue}.tar.gz`);
    }
    return url;
}
function replaceAll(input, from, to) {
    return input.split(from).join(to);
}
function replaceValues(content, from, to) {
    // istanbul ignore if
    if (!from || !to || from === to) {
        return content;
    }
    const massagedFrom = from.replace((0, regex_1.regEx)(/^v/), '');
    const massagedTo = to.replace((0, regex_1.regEx)(/^v/), '');
    return replaceAll(content, massagedFrom, massagedTo);
}
async function getHashFromUrl(url) {
    const cacheNamespace = 'url-sha256';
    const cachedResult = await packageCache.get(cacheNamespace, url);
    /* istanbul ignore next line */
    if (cachedResult) {
        return cachedResult;
    }
    try {
        const hash = await (0, hash_1.hashStream)(http.stream(url), 'sha256');
        const cacheMinutes = 3 * 24 * 60; // 3 days
        await packageCache.set(cacheNamespace, url, hash, cacheMinutes);
        return hash;
    }
    catch /* istanbul ignore next */ {
        return null;
    }
}
async function getHashFromUrls(urls) {
    const hashes = (await (0, promises_1.map)(urls, (url) => getHashFromUrl(massageUrl(url)))).filter(is_1.default.truthy);
    if (!hashes.length) {
        logger_1.logger.debug({ urls }, 'Could not calculate hash for URLs');
        return null;
    }
    const distinctHashes = new Set(hashes);
    // istanbul ignore if
    if (distinctHashes.size > 1) {
        logger_1.logger.warn({ urls }, 'Found multiple hashes for single def');
    }
    return hashes[0];
}
async function updateArtifacts(updateArtifact) {
    const { packageFileName: path, updatedDeps: upgrades } = updateArtifact;
    const oldContents = updateArtifact.newPackageFileContent;
    let newContents = oldContents;
    for (const upgrade of upgrades) {
        const { managerData } = upgrade;
        const idx = managerData?.idx;
        if (upgrade.depType === 'http_file' || upgrade.depType === 'http_archive') {
            const rule = (0, common_1.findCodeFragment)(newContents, [idx]);
            /* v8 ignore start -- used only for type narrowing */
            if (rule?.type !== 'record') {
                continue;
            } /* v8 ignore stop */
            const urlFragments = getUrlFragments(rule);
            if (!urlFragments?.length) {
                logger_1.logger.debug(`def: ${rule.value}, urls is empty`);
                continue;
            }
            const updateValues = (oldUrl) => {
                let url = oldUrl;
                url = replaceValues(url, upgrade.currentValue, upgrade.newValue);
                url = replaceValues(url, upgrade.currentDigest, upgrade.newDigest);
                url = migrateUrl(url, upgrade);
                return url;
            };
            const urls = urlFragments.map(({ value }) => updateValues(value));
            const hash = await getHashFromUrls(urls);
            if (!hash) {
                continue;
            }
            newContents = (0, common_1.patchCodeAtFragments)(newContents, urlFragments, updateValues);
            newContents = (0, common_1.updateCode)(newContents, [idx, 'strip_prefix'], updateValues);
            newContents = (0, common_1.updateCode)(newContents, [idx, 'sha256'], hash);
        }
    }
    if (oldContents === newContents) {
        return null;
    }
    return [
        {
            file: {
                type: 'addition',
                path,
                contents: newContents,
            },
        },
    ];
}
//# sourceMappingURL=artifacts.js.map