"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleToMavenPackageDep = exports.mavenExtensionTags = exports.mavenExtensionPrefix = void 0;
exports.fillRegistryUrls = fillRegistryUrls;
const zod_1 = require("zod");
const maven_1 = require("../../../datasource/maven");
const gradle_1 = require("../../../versioning/gradle");
const fragments_1 = require("./fragments");
const artifactTag = 'artifact';
const installTag = 'install';
const commonDepType = 'maven_install';
exports.mavenExtensionPrefix = 'maven';
exports.mavenExtensionTags = [artifactTag, installTag];
function depTypeByTag(tag) {
    return `maven_${tag}`;
}
const ArtifactSpec = zod_1.z.object({
    group: zod_1.z.string(),
    artifact: zod_1.z.string(),
    version: zod_1.z.string(),
});
const MavenArtifactTarget = fragments_1.ExtensionTagFragmentSchema.extend({
    extension: zod_1.z.literal(exports.mavenExtensionPrefix),
    tag: zod_1.z.literal(artifactTag),
    children: zod_1.z.object({
        artifact: fragments_1.StringFragmentSchema,
        group: fragments_1.StringFragmentSchema,
        version: fragments_1.StringFragmentSchema,
    }),
}).transform(({ children: { artifact, group, version } }) => [
    {
        datasource: maven_1.MavenDatasource.id,
        versioning: gradle_1.id,
        depName: `${group.value}:${artifact.value}`,
        currentValue: version.value,
        depType: depTypeByTag(artifactTag),
    },
]);
const MavenInstallTarget = fragments_1.ExtensionTagFragmentSchema.extend({
    extension: zod_1.z.literal(exports.mavenExtensionPrefix),
    tag: zod_1.z.literal(installTag),
    children: zod_1.z.object({
        artifacts: fragments_1.StringArrayFragmentSchema.transform((artifacts) => {
            const result = [];
            for (const { value } of artifacts.items) {
                const [group, artifact, version] = value.split(':');
                if (group && artifact && version) {
                    result.push({ group, artifact, version });
                }
            }
            return result;
        }),
        repositories: fragments_1.StringArrayFragmentSchema,
    }),
}).transform(({ children: { artifacts, repositories } }) => artifacts.map(({ group, artifact, version: currentValue }) => ({
    datasource: maven_1.MavenDatasource.id,
    versioning: gradle_1.id,
    depName: `${group}:${artifact}`,
    currentValue,
    depType: depTypeByTag(installTag),
    registryUrls: repositories.items.map((i) => i.value),
})));
exports.RuleToMavenPackageDep = zod_1.z.union([
    MavenArtifactTarget,
    MavenInstallTarget,
]);
function fillRegistryUrls(packageDeps) {
    const artifactRules = [];
    const registryUrls = [];
    const result = [];
    // registry urls are specified only in maven.install, not in maven.artifact
    packageDeps.flat().forEach((dep) => {
        if (dep.depType === depTypeByTag(installTag)) {
            if (Array.isArray(dep.registryUrls)) {
                registryUrls.push(...dep.registryUrls);
                result.push(dep);
            }
        }
        else if (dep.depType === depTypeByTag(artifactTag)) {
            artifactRules.push(dep);
        }
    });
    const uniqUrls = [...new Set(registryUrls)];
    for (const artifactRule of artifactRules) {
        artifactRule.registryUrls = uniqUrls;
        artifactRule.depType = commonDepType;
        result.push(artifactRule);
    }
    return result;
}
//# sourceMappingURL=maven.js.map