"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ctx = exports.CtxProcessingError = void 0;
const tslib_1 = require("tslib");
const fragments = tslib_1.__importStar(require("./fragments"));
class CtxProcessingError extends Error {
    current;
    parent;
    constructor(current, parent) {
        const msg = `Invalid context state. current: ${current.type}, parent: ${parent?.type ?? 'none'}`;
        super(msg);
        this.name = 'CtxProcessingError';
        this.current = current;
        this.parent = parent;
    }
}
exports.CtxProcessingError = CtxProcessingError;
class Ctx {
    source;
    results;
    stack;
    constructor(source) {
        this.source = source;
        this.results = [];
        this.stack = [];
    }
    get safeCurrent() {
        return this.stack.at(-1);
    }
    get current() {
        const c = this.safeCurrent;
        if (c === undefined) {
            throw new Error('Requested current, but no value.');
        }
        return c;
    }
    get currentRule() {
        const current = this.current;
        if (current.type === 'rule') {
            return current;
        }
        throw new Error('Requested current rule, but does not exist.');
    }
    get currentExtensionTag() {
        const current = this.current;
        if (current.type === 'extensionTag') {
            return current;
        }
        throw new Error('Requested current extension tag, but does not exist.');
    }
    get currentArray() {
        const current = this.current;
        if (current.type === 'array') {
            return current;
        }
        throw new Error('Requested current array, but does not exist.');
    }
    popPreparedExtensionTag() {
        const c = this.stack.pop();
        if (c === undefined) {
            throw new Error('Requested current, but no value.');
        }
        if (c.type === 'preparedExtensionTag') {
            return c;
        }
        throw new Error('Requested current prepared extension tag, but does not exist.');
    }
    popStack() {
        const current = this.stack.pop();
        if (!current) {
            return false;
        }
        if (!current.isComplete) {
            this.stack.push(current);
            return false;
        }
        const parent = this.safeCurrent;
        if (parent) {
            if (parent.type === 'attribute' && fragments.isValue(current)) {
                parent.value = current;
                parent.isComplete = true;
                return true;
            }
            if (parent.type === 'array' && fragments.isPrimitive(current)) {
                parent.items.push(current);
                return true;
            }
            if ((parent.type === 'rule' || parent.type === 'extensionTag') &&
                current.type === 'attribute' &&
                current.value !== undefined) {
                parent.children[current.name] = current.value;
                return true;
            }
        }
        else if (current.type === 'rule' || current.type === 'extensionTag') {
            this.results.push(current);
            return true;
        }
        throw new CtxProcessingError(current, parent);
    }
    processStack() {
        while (this.popStack()) {
            // Nothing to do
        }
        return this;
    }
    addString(value) {
        this.stack.push(fragments.string(value));
        return this.processStack();
    }
    addBoolean(value) {
        this.stack.push(fragments.boolean(value));
        return this.processStack();
    }
    startRule(name) {
        const rule = fragments.rule(name);
        this.stack.push(rule);
        return this;
    }
    endRule() {
        const rule = this.currentRule;
        rule.isComplete = true;
        return this.processStack();
    }
    prepareExtensionTag(extension, rawExtension, offset) {
        const preppedTag = fragments.preparedExtensionTag(extension, rawExtension, offset);
        this.stack.push(preppedTag);
        return this;
    }
    startExtensionTag(tag) {
        const { extension, rawExtension, offset } = this.popPreparedExtensionTag();
        const extensionTag = fragments.extensionTag(extension, rawExtension, tag, offset);
        this.stack.push(extensionTag);
        return this;
    }
    endExtensionTag(offset) {
        const tag = this.currentExtensionTag;
        tag.isComplete = true;
        tag.rawString = this.source.slice(tag.offset, offset);
        return this.processStack();
    }
    startAttribute(name) {
        this.stack.push(fragments.attribute(name));
        return this.processStack();
    }
    startArray() {
        this.stack.push(fragments.array());
        return this.processStack();
    }
    endArray() {
        const array = this.currentArray;
        array.isComplete = true;
        return this.processStack();
    }
}
exports.Ctx = Ctx;
//# sourceMappingURL=context.js.map