"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const regex_1 = require("../../../util/regex");
const schema_utils_1 = require("../../../util/schema-utils");
const url_1 = require("../../../util/url");
const docker_1 = require("../../datasource/docker");
const git_tags_1 = require("../../datasource/git-tags");
const helm_1 = require("../../datasource/helm");
const extract_1 = require("../dockerfile/extract");
const oci_1 = require("../helmv3/oci");
const schema_1 = require("./schema");
const util_1 = require("./util");
const kustomizeImageRe = (0, regex_1.regEx)(/=(?<image>.+)$/);
function extractPackageFile(content, packageFile, _config) {
    // check for argo reference. API version for the kind attribute is used
    if (util_1.fileTestRegex.test(content) === false) {
        logger_1.logger.debug(`Skip file ${packageFile} as no argoproj.io apiVersion could be found in matched file`);
        return null;
    }
    const definitions = schema_1.ApplicationDefinitionSchema.catch((0, schema_utils_1.withDebugMessage)([], `${packageFile} does not match schema`)).parse(content);
    const deps = definitions.flatMap(processAppSpec);
    return deps.length ? { deps } : null;
}
function processSource(source) {
    // a chart variable is defined this is helm declaration
    if (source.chart) {
        // assume OCI helm chart if repoURL doesn't contain explicit protocol
        if ((0, oci_1.isOCIRegistry)(source.repoURL) || !source.repoURL.includes('://')) {
            const registryURL = (0, url_1.trimTrailingSlash)((0, oci_1.removeOCIPrefix)(source.repoURL));
            return [
                {
                    depName: `${registryURL}/${source.chart}`,
                    currentValue: source.targetRevision,
                    datasource: docker_1.DockerDatasource.id,
                },
            ];
        }
        return [
            {
                depName: source.chart,
                registryUrls: [source.repoURL],
                currentValue: source.targetRevision,
                datasource: helm_1.HelmDatasource.id,
            },
        ];
    }
    const dependencies = [
        {
            depName: source.repoURL,
            currentValue: source.targetRevision,
            datasource: git_tags_1.GitTagsDatasource.id,
        },
    ];
    // Git repo is pointing to a Kustomize resources
    if (source.kustomize?.images) {
        dependencies.push(...source.kustomize.images.map(processKustomizeImage).filter(is_1.default.truthy));
    }
    return dependencies;
}
function processAppSpec(definition) {
    const spec = definition.kind === 'Application'
        ? definition.spec
        : definition.spec.template.spec;
    const deps = [];
    if (is_1.default.nonEmptyObject(spec.source)) {
        deps.push(...processSource(spec.source));
    }
    for (const source of (0, array_1.coerceArray)(spec.sources)) {
        deps.push(...processSource(source));
    }
    return deps;
}
function processKustomizeImage(kustomizeImage) {
    const parts = kustomizeImageRe.exec(kustomizeImage);
    if (!parts?.groups?.image) {
        return null;
    }
    return (0, extract_1.getDep)(parts.groups.image);
}
//# sourceMappingURL=extract.js.map