"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Unity3dDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const timestamp_1 = require("../../../util/timestamp");
const Unity3dVersioning = tslib_1.__importStar(require("../../versioning/unity3d"));
const datasource_1 = require("../datasource");
const schema_1 = require("./schema");
class Unity3dDatasource extends datasource_1.Datasource {
    static baseUrl = 'https://services.api.unity.com/unity/editor/release/v1/releases';
    static homepage = 'https://unity.com/';
    static streams = {
        lts: `${Unity3dDatasource.baseUrl}?stream=LTS`,
        tech: `${Unity3dDatasource.baseUrl}?stream=TECH`,
        alpha: `${Unity3dDatasource.baseUrl}?stream=ALPHA`,
        beta: `${Unity3dDatasource.baseUrl}?stream=BETA`,
    };
    static legacyStreams = {
        lts: `${Unity3dDatasource.homepage}releases/editor/lts-releases.xml`,
        stable: `${Unity3dDatasource.homepage}releases/editor/releases.xml`,
        beta: `${Unity3dDatasource.homepage}releases/editor/beta/latest.xml`,
    };
    static limit = 25;
    static id = 'unity3d';
    defaultRegistryUrls = [Unity3dDatasource.streams.lts];
    defaultVersioning = Unity3dVersioning.id;
    registryStrategy = 'merge';
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `releaseDate` field in the results.';
    constructor() {
        super(Unity3dDatasource.id);
    }
    translateStream(registryUrl) {
        const legacyKey = Object.keys(Unity3dDatasource.legacyStreams).find((key) => Unity3dDatasource.legacyStreams[key] === registryUrl);
        if (legacyKey) {
            if (legacyKey === 'stable') {
                return Unity3dDatasource.streams.lts;
            }
            return Unity3dDatasource.streams[legacyKey];
        }
        return registryUrl;
    }
    async getByStream(registryUrl, withHash) {
        const translatedRegistryUrl = this.translateStream(registryUrl);
        const isStable = translatedRegistryUrl === Unity3dDatasource.streams.lts;
        let total = null;
        const result = {
            releases: [],
            homepage: Unity3dDatasource.homepage,
            registryUrl: translatedRegistryUrl,
        };
        for (let offset = 0; total === null || offset < total; offset += Unity3dDatasource.limit) {
            const response = await this.http.getJson(`${translatedRegistryUrl}&limit=${Unity3dDatasource.limit}&offset=${offset}`, schema_1.UnityReleasesJSON);
            for (const release of response.body.results) {
                result.releases.push({
                    version: withHash
                        ? `${release.version} (${release.shortRevision})`
                        : release.version,
                    releaseTimestamp: (0, timestamp_1.asTimestamp)(release.releaseDate),
                    changelogUrl: release.releaseNotes.url,
                    isStable,
                });
            }
            total ??= response.body.total;
        }
        return result;
    }
    async getReleases({ packageName, registryUrl, }) {
        return await this.getByStream(registryUrl, packageName === 'm_EditorVersionWithRevision');
    }
}
exports.Unity3dDatasource = Unity3dDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${Unity3dDatasource.id}`,
        key: ({ registryUrl, packageName }) => `${registryUrl}:${packageName}`,
    })
], Unity3dDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map