"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RubyVersionDatasource = void 0;
const tslib_1 = require("tslib");
const logger_1 = require("../../../logger");
const external_host_error_1 = require("../../../types/errors/external-host-error");
const decorator_1 = require("../../../util/cache/package/decorator");
const html_1 = require("../../../util/html");
const timestamp_1 = require("../../../util/timestamp");
const ruby_1 = require("../../versioning/ruby");
const datasource_1 = require("../datasource");
class RubyVersionDatasource extends datasource_1.Datasource {
    static id = 'ruby-version';
    constructor() {
        super(RubyVersionDatasource.id);
    }
    defaultRegistryUrls = ['https://www.ruby-lang.org/'];
    customRegistrySupport = false;
    defaultVersioning = ruby_1.id;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `release-list` table in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'We use the URL: https://github.com/ruby/ruby.';
    async getReleases({ registryUrl, }) {
        const res = {
            homepage: 'https://www.ruby-lang.org',
            sourceUrl: 'https://github.com/ruby/ruby',
            releases: [],
        };
        // TODO: types (#22198)
        const rubyVersionsUrl = `${registryUrl}en/downloads/releases/`;
        try {
            const response = await this.http.getText(rubyVersionsUrl);
            const root = (0, html_1.parse)(response.body);
            const rows = root.querySelector('.release-list')?.querySelectorAll('tr') ?? [];
            rows.forEach((row) => {
                const tds = row.querySelectorAll('td');
                const columns = [];
                tds.forEach((td) => columns.push(td.innerHTML));
                if (columns.length) {
                    const version = columns[0].replace('Ruby ', '');
                    if ((0, ruby_1.isVersion)(version)) {
                        const releaseTimestamp = (0, timestamp_1.asTimestamp)(columns[1]);
                        const changelogUrl = columns[2]
                            .replace('<a href="', 'https://www.ruby-lang.org')
                            .replace('">more...</a>', '');
                        res.releases.push({ version, releaseTimestamp, changelogUrl });
                    }
                }
            });
            if (!res.releases.length) {
                logger_1.logger.warn({ registryUrl }, 'Missing ruby releases');
                return null;
            }
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        return res;
    }
    handleHttpErrors(err) {
        throw new external_host_error_1.ExternalHostError(err);
    }
}
exports.RubyVersionDatasource = RubyVersionDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({ namespace: `datasource-${RubyVersionDatasource.id}`, key: 'all' })
], RubyVersionDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map