"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.massageUrl = massageUrl;
exports.massageGithubUrl = massageGithubUrl;
exports.addMetaData = addMetaData;
exports.shouldDeleteHomepage = shouldDeleteHomepage;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const github_url_from_git_1 = tslib_1.__importDefault(require("github-url-from-git"));
const common_1 = require("../../util/common");
const url_1 = require("../../util/git/url");
const hostRules = tslib_1.__importStar(require("../../util/host-rules"));
const regex_1 = require("../../util/regex");
const timestamp_1 = require("../../util/timestamp");
const url_2 = require("../../util/url");
const metadata_manual_1 = require("./metadata-manual");
const githubPages = (0, regex_1.regEx)('^https://([^.]+).github.com/([^/]+)$');
const gitPrefix = (0, regex_1.regEx)('^git:/?/?');
function massageUrl(sourceUrl) {
    // Replace git@ sourceUrl with https so hostname can be parsed
    const massagedUrl = massageGitAtUrl(sourceUrl);
    // Check if URL is valid
    const parsedUrl = (0, url_2.parseUrl)(massagedUrl);
    if (!parsedUrl) {
        return '';
    }
    if ((0, common_1.detectPlatform)(massagedUrl) === 'gitlab') {
        return massageGitlabUrl(sourceUrl);
    }
    return massageGithubUrl(sourceUrl);
}
function massageGithubUrl(url) {
    const massagedUrl = massageGitAtUrl(url);
    return massagedUrl
        .replace('http:', 'https:')
        .replace('http+git:', 'https:')
        .replace('https+git:', 'https:')
        .replace('ssh://git@', 'https://')
        .replace(gitPrefix, 'https://')
        .replace(githubPages, 'https://github.com/$1/$2')
        .replace('www.github.com', 'github.com')
        .split('/')
        .slice(0, 5)
        .join('/');
}
function massageGitlabUrl(url) {
    const massagedUrl = massageGitAtUrl(url);
    return massagedUrl
        .replace('http:', 'https:')
        .replace(gitPrefix, 'https://')
        .replace((0, regex_1.regEx)(/\/tree\/.*$/i), '')
        .replace((0, regex_1.regEx)(/\/$/i), '')
        .replace('.git', '');
}
function massageGitAtUrl(url) {
    let massagedUrl = url;
    if (url.startsWith('git@')) {
        massagedUrl = url.replace(':', '/').replace('git@', 'https://');
    }
    return massagedUrl;
}
function massageTimestamps(dep) {
    for (const release of dep.releases || []) {
        let { releaseTimestamp } = release;
        delete release.releaseTimestamp;
        releaseTimestamp = (0, timestamp_1.asTimestamp)(releaseTimestamp);
        if (releaseTimestamp) {
            release.releaseTimestamp = releaseTimestamp;
        }
    }
}
function addMetaData(dep, datasource, packageName) {
    massageTimestamps(dep);
    const packageNameLowercase = packageName.toLowerCase();
    const manualChangelogUrl = metadata_manual_1.manualChangelogUrls[datasource]?.[packageNameLowercase];
    if (manualChangelogUrl) {
        dep.changelogUrl = manualChangelogUrl;
    }
    const manualSourceUrl = metadata_manual_1.manualSourceUrls[datasource]?.[packageNameLowercase];
    if (manualSourceUrl) {
        dep.sourceUrl = manualSourceUrl;
    }
    if (dep.sourceUrl && !dep.sourceDirectory) {
        try {
            const parsed = (0, url_1.parseGitUrl)(dep.sourceUrl);
            if (parsed.filepathtype === 'tree' && parsed.filepath !== '') {
                dep.sourceUrl = parsed.toString();
                dep.sourceDirectory = parsed.filepath;
            }
        }
        catch {
            // ignore invalid urls
        }
    }
    if (!dep.sourceUrl &&
        dep.changelogUrl &&
        (0, common_1.detectPlatform)(dep.changelogUrl) === 'github') {
        dep.sourceUrl = dep.changelogUrl;
    }
    if (!dep.sourceUrl && dep.homepage) {
        const platform = (0, common_1.detectPlatform)(dep.homepage);
        if (platform === 'github' || platform === 'gitlab') {
            dep.sourceUrl = dep.homepage;
        }
    }
    const extraBaseUrls = [];
    // istanbul ignore next
    hostRules.hosts({ hostType: 'github' }).forEach((host) => {
        extraBaseUrls.push(host, `gist.${host}`);
    });
    extraBaseUrls.push('gitlab.com');
    if (dep.sourceUrl) {
        // try massaging it
        const massagedUrl = massageUrl(dep.sourceUrl);
        if (is_1.default.emptyString(massagedUrl)) {
            delete dep.sourceUrl;
        }
        else {
            // parse from github-url-from-git only supports Github URLs as its name implies
            dep.sourceUrl =
                (0, github_url_from_git_1.default)(massagedUrl, {
                    extraBaseUrls,
                }) || dep.sourceUrl;
            // prefer massaged URL to source URL if the latter does not start with http:// or https://
            // (e.g. git@somehost.com) and the detected platform is gitlab.
            // this allows to retrieve changelogs from git hosts other than Github
            if (!(0, url_2.isHttpUrl)(dep.sourceUrl) &&
                (0, common_1.detectPlatform)(massagedUrl) === 'gitlab') {
                dep.sourceUrl = massagedUrl;
            }
        }
    }
    if (shouldDeleteHomepage(dep.sourceUrl, dep.homepage)) {
        delete dep.homepage;
    }
    // Clean up any empty urls
    const urlKeys = [
        'homepage',
        'sourceUrl',
        'changelogUrl',
        'dependencyUrl',
    ];
    for (const urlKey of urlKeys) {
        const urlVal = dep[urlKey];
        if (is_1.default.string(urlVal) && (0, url_2.isHttpUrl)(urlVal.trim())) {
            dep[urlKey] = urlVal.trim();
        }
        else {
            delete dep[urlKey];
        }
    }
}
/**
 * Returns true if
 * 1. it's a github or gitlab url and not a path within the repo.
 * 2. it's equal to sourceURl
 * @param sourceUrl
 * @param homepage
 */
function shouldDeleteHomepage(sourceUrl, homepage) {
    if (is_1.default.nullOrUndefined(sourceUrl) || is_1.default.undefined(homepage)) {
        return false;
    }
    const massagedSourceUrl = massageUrl(sourceUrl);
    const platform = (0, common_1.detectPlatform)(homepage);
    if (platform === 'github' || platform === 'gitlab') {
        const sourceUrlParsed = (0, url_2.parseUrl)(massagedSourceUrl);
        if (is_1.default.nullOrUndefined(sourceUrlParsed)) {
            return false;
        }
        const homepageParsed = (0, url_2.parseUrl)(homepage);
        return (homepageParsed !== null &&
            (0, url_2.trimTrailingSlash)(homepageParsed.pathname) ===
                (0, url_2.trimTrailingSlash)(sourceUrlParsed.pathname));
    }
    return massagedSourceUrl === homepage;
}
//# sourceMappingURL=metadata.js.map