"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MavenDatasource = exports.defaultRegistryUrls = void 0;
const tslib_1 = require("tslib");
const logger_1 = require("../../../logger");
const packageCache = tslib_1.__importStar(require("../../../util/cache/package"));
const timestamp_1 = require("../../../util/timestamp");
const url_1 = require("../../../util/url");
const maven_1 = tslib_1.__importDefault(require("../../versioning/maven"));
const mavenVersioning = tslib_1.__importStar(require("../../versioning/maven"));
const compare_1 = require("../../versioning/maven/compare");
const datasource_1 = require("../datasource");
const common_1 = require("./common");
const util_1 = require("./util");
function getLatestSuitableVersion(releases) {
    /* v8 ignore next 3 -- TODO: add test */
    if (!releases?.length) {
        return null;
    }
    const allVersions = releases.map(({ version }) => version);
    const stableVersions = allVersions.filter((x) => maven_1.default.isStable(x));
    const versions = stableVersions.length ? stableVersions : allVersions;
    return versions.reduce((latestVersion, version) => (0, compare_1.compare)(version, latestVersion) === 1
        ? version
        : /* istanbul ignore next: hard to test */ latestVersion);
}
function extractVersions(metadata) {
    const res = {};
    const versions = metadata.descendantWithPath('versioning.versions');
    const elements = versions?.childrenNamed('version');
    if (!elements) {
        return res;
    }
    res.versions = elements.map((el) => el.val);
    const latest = metadata.descendantWithPath('versioning.latest');
    if (latest?.val) {
        res.tags ??= {};
        res.tags.latest = latest.val;
    }
    const release = metadata.descendantWithPath('versioning.release');
    if (release?.val) {
        res.tags ??= {};
        res.tags.release = release.val;
    }
    return res;
}
exports.defaultRegistryUrls = [common_1.MAVEN_REPO];
class MavenDatasource extends datasource_1.Datasource {
    static id = 'maven';
    caching = true;
    defaultRegistryUrls = exports.defaultRegistryUrls;
    defaultVersioning = mavenVersioning.id;
    registryStrategy = 'merge';
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `Last-Modified` header or the `lastModified` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined from the `scm` tags in the results.';
    constructor(id = MavenDatasource.id) {
        super(id);
    }
    async fetchVersionsFromMetadata(dependency, repoUrl) {
        const metadataUrl = (0, util_1.getMavenUrl)(dependency, repoUrl, 'maven-metadata.xml');
        const metadataXmlResult = await (0, util_1.downloadMavenXml)(this.http, metadataUrl);
        return metadataXmlResult
            .transform(({ data: metadata }) => extractVersions(metadata))
            .onError((err) => {
            logger_1.logger.debug(`Maven: error fetching versions for "${dependency.display}": ${err.type}`);
        })
            .unwrapOr({});
    }
    async getReleases({ packageName, registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const dependency = (0, util_1.getDependencyParts)(packageName);
        const repoUrl = (0, url_1.ensureTrailingSlash)(registryUrl);
        logger_1.logger.debug(`Looking up ${dependency.display} in repository ${repoUrl}`);
        const metadata = await this.fetchVersionsFromMetadata(dependency, repoUrl);
        if (!metadata.versions?.length) {
            return null;
        }
        const releases = metadata.versions.map((version) => ({ version }));
        logger_1.logger.debug(`Found ${releases.length} new releases for ${dependency.display} in repository ${repoUrl}`);
        const latestSuitableVersion = getLatestSuitableVersion(releases);
        const dependencyInfo = latestSuitableVersion &&
            (await (0, util_1.getDependencyInfo)(this.http, dependency, repoUrl, latestSuitableVersion));
        const result = {
            ...dependency,
            ...dependencyInfo,
            releases,
        };
        if (metadata.tags) {
            result.tags = metadata.tags;
        }
        if (!this.defaultRegistryUrls.includes(registryUrl)) {
            result.isPrivate = true;
        }
        return result;
    }
    async postprocessRelease({ packageName, registryUrl }, release) {
        const { version, versionOrig } = release;
        const cacheKey = versionOrig
            ? `postprocessRelease:${registryUrl}:${packageName}:${versionOrig}:${version}`
            : `postprocessRelease:${registryUrl}:${packageName}:${version}`;
        const cachedResult = await packageCache.get('datasource-maven:postprocess-reject', cacheKey);
        /* v8 ignore start: hard to test */
        if (cachedResult) {
            return cachedResult;
        } /* v8 ignore stop */
        if (!packageName || !registryUrl) {
            return release;
        }
        const dependency = (0, util_1.getDependencyParts)(packageName);
        const pomUrl = await (0, util_1.createUrlForDependencyPom)(this.http, release.versionOrig ?? release.version, dependency, registryUrl);
        const artifactUrl = (0, util_1.getMavenUrl)(dependency, registryUrl, pomUrl);
        const fetchResult = await (0, util_1.downloadMaven)(this.http, artifactUrl);
        const { val, err } = fetchResult.unwrap();
        if (err) {
            const result = err.type === 'not-found' ? 'reject' : release;
            if (result === 'reject') {
                await packageCache.set('datasource-maven:postprocess-reject', cacheKey, result, 24 * 60);
            }
            return result;
        }
        if (val.lastModified) {
            release.releaseTimestamp = (0, timestamp_1.asTimestamp)(val.lastModified);
        }
        return release;
    }
}
exports.MavenDatasource = MavenDatasource;
//# sourceMappingURL=index.js.map