"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GoDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const decorator_1 = require("../../../util/cache/package/decorator");
const env_1 = require("../../../util/env");
const regex_1 = require("../../../util/regex");
const sanitize_1 = require("../../../util/sanitize");
const url_1 = require("../../../util/url");
const semver_1 = require("../../versioning/semver");
const bitbucket_tags_1 = require("../bitbucket-tags");
const datasource_1 = require("../datasource");
const git_tags_1 = require("../git-tags");
const gitea_tags_1 = require("../gitea-tags");
const github_tags_1 = require("../github-tags");
const gitlab_tags_1 = require("../gitlab-tags");
const base_1 = require("./base");
const goproxy_parser_1 = require("./goproxy-parser");
const releases_direct_1 = require("./releases-direct");
const releases_goproxy_1 = require("./releases-goproxy");
class GoDatasource extends datasource_1.Datasource {
    static id = 'go';
    defaultVersioning = semver_1.id;
    constructor() {
        super(GoDatasource.id);
    }
    defaultConfig = {
        commitMessageTopic: 'module {{depName}}',
    };
    customRegistrySupport = false;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'If the release timestamp is not returned from the respective datasoure used to fetch the releases, then Renovate uses the `Time` field in the results instead.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined from the `packageName` and `registryUrl`.';
    goproxy = new releases_goproxy_1.GoProxyDatasource();
    direct = new releases_direct_1.GoDirectDatasource();
    // Pseudo versions https://go.dev/ref/mod#pseudo-versions
    static pversionRegexp = (0, regex_1.regEx)(/v\d+\.\d+\.\d+-(?:\w+\.)?(?:0\.)?\d{14}-(?<digest>[a-f0-9]{12})/);
    getReleases(config) {
        return this.goproxy.getReleases(config);
    }
    /**
     * go.getDigest
     *
     * This datasource resolves a go module URL into its source repository
     *  and then fetches the digest if it is on GitHub.
     *
     * This function will:
     *  - Determine the source URL for the module
     *  - Call the respective getDigest in github to retrieve the commit hash
     */
    async getDigest({ packageName }, newValue) {
        if ((0, goproxy_parser_1.parseGoproxy)().some(({ url }) => url === 'off')) {
            logger_1.logger.debug(`Skip digest fetch for ${packageName} with GOPROXY containing "off"`);
            return null;
        }
        const source = await base_1.BaseGoDatasource.getDatasource(packageName);
        if (!source) {
            return null;
        }
        // ignore vX.Y.Z-(0.)? pseudo versions that are used Go Modules - look up default branch instead
        // ignore v0.0.0 versions to fetch the digest of default branch, not the commit of non-existing tag `v0.0.0`
        const tag = newValue &&
            !GoDatasource.pversionRegexp.test(newValue) &&
            newValue !== 'v0.0.0'
            ? newValue
            : undefined;
        switch (source.datasource) {
            case git_tags_1.GitTagsDatasource.id: {
                return this.direct.git.getDigest(source, tag);
            }
            case gitea_tags_1.GiteaTagsDatasource.id: {
                return this.direct.gitea.getDigest(source, tag);
            }
            case github_tags_1.GithubTagsDatasource.id: {
                return this.direct.github.getDigest(source, tag);
            }
            case bitbucket_tags_1.BitbucketTagsDatasource.id: {
                return this.direct.bitbucket.getDigest(source, tag);
            }
            case gitlab_tags_1.GitlabTagsDatasource.id: {
                return this.direct.gitlab.getDigest(source, tag);
            }
            /* istanbul ignore next: can never happen, makes lint happy */
            default: {
                return null;
            }
        }
    }
}
exports.GoDatasource = GoDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${GoDatasource.id}`,
        // TODO: types (#22198)
        key: ({ packageName }) => `getReleases:${packageName}`,
    })
], GoDatasource.prototype, "getReleases", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${GoDatasource.id}`,
        key: ({ packageName }, newValue) => `getDigest:${packageName}:${newValue}`,
    })
], GoDatasource.prototype, "getDigest", null);
const env = (0, env_1.getEnv)();
/* v8 ignore next 3 -- hard to test */
if (is_1.default.string(env.GOPROXY)) {
    const uri = (0, url_1.parseUrl)(env.GOPROXY);
    if (uri?.password) {
        (0, sanitize_1.addSecretForSanitizing)(uri.password, 'global');
    }
    else if (uri?.username) {
        (0, sanitize_1.addSecretForSanitizing)(uri.username, 'global');
    }
}
//# sourceMappingURL=index.js.map