"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseGoDatasource = void 0;
const tslib_1 = require("tslib");
// TODO: types (#22198)
const global_1 = require("../../../config/global");
const logger_1 = require("../../../logger");
const common_1 = require("../../../util/common");
const hostRules = tslib_1.__importStar(require("../../../util/host-rules"));
const http_1 = require("../../../util/http");
const regex_1 = require("../../../util/regex");
const url_1 = require("../../../util/url");
const bitbucket_tags_1 = require("../bitbucket-tags");
const git_tags_1 = require("../git-tags");
const gitea_tags_1 = require("../gitea-tags");
const github_tags_1 = require("../github-tags");
const gitlab_tags_1 = require("../gitlab-tags");
// TODO: figure out class hierarchy (#10532)
class BaseGoDatasource {
    static gitlabHttpsRegExp = (0, regex_1.regEx)(/^(?<httpsRegExpUrl>https:\/\/[^/]*gitlab\.[^/]*)\/(?<httpsRegExpName>.+?)(?:\/v\d+)?[/]?$/);
    static gitlabRegExp = (0, regex_1.regEx)(/^(?<regExpUrl>gitlab\.[^/]*)\/(?<regExpPath>.+?)(?:\/v\d+)?[/]?$/);
    static gitVcsRegexp = (0, regex_1.regEx)(/^(?:[^/]+)\/(?<module>.*)\.git(?:$|\/)/);
    static id = 'go';
    static http = new http_1.Http(BaseGoDatasource.id);
    static async getDatasource(goModule) {
        if (goModule.startsWith('gopkg.in/')) {
            const [pkg] = goModule.replace('gopkg.in/', '').split('.');
            const packageName = pkg.includes('/') ? pkg : `go-${pkg}/${pkg}`;
            return {
                datasource: github_tags_1.GithubTagsDatasource.id,
                packageName,
                registryUrl: 'https://github.com',
            };
        }
        if (goModule.startsWith('github.com/')) {
            const split = goModule.split('/');
            const packageName = split[1] + '/' + split[2];
            return {
                datasource: github_tags_1.GithubTagsDatasource.id,
                packageName,
                registryUrl: 'https://github.com',
            };
        }
        if (goModule.startsWith('bitbucket.org/')) {
            const split = goModule.split('/');
            const packageName = split[1] + '/' + split[2];
            return {
                datasource: bitbucket_tags_1.BitbucketTagsDatasource.id,
                packageName,
                registryUrl: 'https://bitbucket.org',
            };
        }
        if (goModule.startsWith('code.cloudfoundry.org/')) {
            const packageName = goModule.replace('code.cloudfoundry.org', 'cloudfoundry');
            return {
                datasource: github_tags_1.GithubTagsDatasource.id,
                packageName,
                registryUrl: 'https://github.com',
            };
        }
        if (goModule.startsWith('dev.azure.com/')) {
            const split = goModule.split('/');
            if ((split.length > 4 && split[3] === '_git') || split.length > 3) {
                const packageName = 'https://dev.azure.com/' +
                    split[1] +
                    '/' +
                    split[2] +
                    '/_git/' +
                    (split[3] === '_git' ? split[4] : split[3]).replace((0, regex_1.regEx)(/\.git$/), '');
                return {
                    datasource: git_tags_1.GitTagsDatasource.id,
                    packageName,
                };
            }
        }
        //#region known gitea compatible hosts
        if (goModule.startsWith('gitea.com/')) {
            const split = goModule.split('/');
            const packageName = `${split[1]}/${split[2]}`;
            return {
                datasource: gitea_tags_1.GiteaTagsDatasource.id,
                packageName,
                registryUrl: 'https://gitea.com',
            };
        }
        if (goModule.startsWith('code.forgejo.org/')) {
            const split = goModule.split('/');
            const packageName = `${split[1]}/${split[2]}`;
            return {
                datasource: gitea_tags_1.GiteaTagsDatasource.id,
                packageName,
                registryUrl: 'https://code.forgejo.org',
            };
        }
        if (goModule.startsWith('codeberg.org/')) {
            const split = goModule.split('/');
            const packageName = `${split[1]}/${split[2]}`;
            return {
                datasource: gitea_tags_1.GiteaTagsDatasource.id,
                packageName,
                registryUrl: 'https://codeberg.org',
            };
        }
        //#endregion
        return await BaseGoDatasource.goGetDatasource(goModule);
    }
    static async goGetDatasource(goModule) {
        const goModuleUrl = goModule.replace(/\.git(\/[a-z0-9/]*)?$/, '');
        const pkgUrl = `https://${goModuleUrl}?go-get=1`;
        const { body: html } = await BaseGoDatasource.http.getText(pkgUrl);
        const goSourceHeader = BaseGoDatasource.goSourceHeader(html, goModule);
        if (goSourceHeader) {
            return goSourceHeader;
        }
        // GitHub Enterprise only returns a go-import meta
        const goImport = BaseGoDatasource.goImportHeader(html, goModule);
        if (goImport) {
            return goImport;
        }
        logger_1.logger.trace({ goModule }, 'No go-source or go-import header found');
        return null;
    }
    static goSourceHeader(html, goModule) {
        const sourceMatchGroups = (0, regex_1.regEx)(/<meta\s+name="?go-source"?\s+content="(?<prefix>[^"\s]+)\s+(?<goSourceUrl>[^"\s]+)/).exec(html)?.groups;
        if (!sourceMatchGroups) {
            return null;
        }
        const { prefix, goSourceUrl } = sourceMatchGroups;
        if (!goModule.startsWith(prefix)) {
            logger_1.logger.trace({ goModule }, 'go-source header prefix not match');
            return null;
        }
        logger_1.logger.debug(`Go lookup source url ${goSourceUrl} for module ${goModule}`);
        return this.detectDatasource(goSourceUrl, goModule);
    }
    static detectDatasource(metadataUrl, goModule) {
        if (metadataUrl.startsWith('https://github.com/')) {
            return {
                datasource: github_tags_1.GithubTagsDatasource.id,
                packageName: metadataUrl
                    .replace('https://github.com/', '')
                    .replace((0, regex_1.regEx)(/\/$/), ''),
                registryUrl: 'https://github.com',
            };
        }
        const gitlabModuleName = BaseGoDatasource.gitlabRegExp.exec(goModule)?.groups?.regExpPath;
        const vcsIndicatedModule = BaseGoDatasource.gitVcsRegexp.exec(goModule)?.groups?.module;
        const metadataUrlMatchGroups = BaseGoDatasource.gitlabHttpsRegExp.exec(metadataUrl)?.groups;
        if (metadataUrlMatchGroups) {
            const { httpsRegExpUrl, httpsRegExpName } = metadataUrlMatchGroups;
            const packageName = vcsIndicatedModule ?? gitlabModuleName ?? httpsRegExpName;
            return {
                datasource: gitlab_tags_1.GitlabTagsDatasource.id,
                registryUrl: httpsRegExpUrl,
                packageName,
            };
        }
        if (hostRules.hostType({ url: metadataUrl }) === 'gitlab') {
            const parsedUrl = (0, url_1.parseUrl)(metadataUrl);
            if (!parsedUrl) {
                logger_1.logger.trace({ goModule }, 'Could not parse go-source URL');
                return null;
            }
            const endpoint = global_1.GlobalConfig.get('endpoint', '');
            const endpointPrefix = (0, regex_1.regEx)(/https:\/\/[^/]+\/(?<prefix>.*?\/)(?:api\/v4\/?)?/).exec(endpoint)?.groups?.prefix;
            let packageName = 
            // a .git path indicates a concrete git repository, which can be different from metadata returned by gitlab
            vcsIndicatedModule ?? (0, url_1.trimLeadingSlash)(parsedUrl.pathname);
            if (endpointPrefix && endpointPrefix !== 'api/') {
                packageName = packageName.replace(endpointPrefix, '');
            }
            const registryUrl = endpointPrefix
                ? endpoint.replace((0, regex_1.regEx)(/\/api\/v4\/?$/), '/')
                : `${parsedUrl.protocol}//${parsedUrl.host}`;
            return {
                datasource: gitlab_tags_1.GitlabTagsDatasource.id,
                registryUrl,
                packageName,
            };
        }
        /* istanbul ignore next */
        return null;
    }
    static goImportHeader(html, goModule) {
        const importMatchGroups = (0, regex_1.regEx)(/<meta\s+name="?go-import"?\s+content="(?<prefix>[^"\s]+)\s+(?<proto>[^"\s]+)\s+(?<goImportURL>[^"\s]+)/).exec(html)?.groups;
        if (!importMatchGroups) {
            logger_1.logger.trace({ goModule }, 'No go-source or go-import header found');
            return null;
        }
        const { prefix, proto, goImportURL } = importMatchGroups;
        if (!goModule.startsWith(prefix)) {
            logger_1.logger.trace({ goModule }, 'go-import header prefix not match');
            return null;
        }
        if (proto !== 'git') {
            logger_1.logger.trace({ goModule }, 'go-import header proto not git');
            return null;
        }
        // get server base url from import url
        const parsedUrl = (0, url_1.parseUrl)(goImportURL);
        if (!parsedUrl) {
            logger_1.logger.trace({ goModule }, 'Could not parse go-import URL');
            return null;
        }
        logger_1.logger.debug(`Go module: ${goModule} lookup import url ${goImportURL}`);
        const datasource = this.detectDatasource(goImportURL.replace((0, regex_1.regEx)(/\.git$/), ''), goModule);
        if (datasource !== null) {
            return datasource;
        }
        // fall back to old behavior if detection did not work
        switch ((0, common_1.detectPlatform)(goImportURL)) {
            case 'github': {
                // split the go module from the URL: host/go/module -> go/module
                // TODO: `parsedUrl.pathname` can be undefined
                const packageName = (0, url_1.trimTrailingSlash)(`${parsedUrl.pathname}`)
                    .replace((0, regex_1.regEx)(/\.git$/), '')
                    .split('/')
                    .slice(-2)
                    .join('/');
                return {
                    datasource: github_tags_1.GithubTagsDatasource.id,
                    registryUrl: `${parsedUrl.protocol}//${parsedUrl.host}`,
                    packageName,
                };
            }
            case 'azure': {
                return {
                    datasource: git_tags_1.GitTagsDatasource.id,
                    packageName: goImportURL.replace((0, regex_1.regEx)(/\.git$/), ''),
                };
            }
            default: {
                return {
                    datasource: git_tags_1.GitTagsDatasource.id,
                    packageName: goImportURL,
                };
            }
        }
    }
}
exports.BaseGoDatasource = BaseGoDatasource;
//# sourceMappingURL=base.js.map