"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubRunnersDatasource = void 0;
const docker_1 = require("../../versioning/docker");
const datasource_1 = require("../datasource");
class GithubRunnersDatasource extends datasource_1.Datasource {
    static id = 'github-runners';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'We use the URL: https://github.com/actions/runner-images.';
    /**
     * Unstable runners must have the `isStable: false` property.
     * Deprecated runners must have the `isDeprecated: true` property.
     * Stable runners should have no extra properties.
     * For more details, read the github-runners datasource readme.
     * Check https://github.blog/changelog/label/actions/ for stable and deprecation dates.
     */
    static releases = {
        ubuntu: [
            { version: '24.04' },
            { version: '24.04-arm', isStable: false },
            { version: '22.04' },
            { version: '22.04-arm', isStable: false },
            { version: '20.04', isDeprecated: true },
            { version: '18.04', isDeprecated: true },
            { version: '16.04', isDeprecated: true },
        ],
        macos: [
            { version: '15', isStable: false },
            { version: '15-large', isStable: false },
            { version: '15-xlarge', isStable: false },
            { version: '14' },
            { version: '14-large' },
            { version: '14-xlarge' },
            { version: '13' },
            { version: '13-large' },
            { version: '13-xlarge' },
            { version: '12-large', isDeprecated: true },
            { version: '12', isDeprecated: true },
            { version: '11', isDeprecated: true },
            { version: '10.15', isDeprecated: true },
        ],
        windows: [
            { version: '2025' },
            { version: '2022' },
            { version: '2019', isDeprecated: true },
            { version: '2016', isDeprecated: true },
        ],
    };
    static isValidRunner(runnerName, runnerVersion) {
        const runnerReleases = GithubRunnersDatasource.releases[runnerName];
        if (!runnerReleases) {
            return false;
        }
        const versionExists = runnerReleases.some(({ version }) => version === runnerVersion);
        return runnerVersion === 'latest' || versionExists;
    }
    defaultVersioning = docker_1.id;
    constructor() {
        super(GithubRunnersDatasource.id);
    }
    getReleases({ packageName, }) {
        const releases = GithubRunnersDatasource.releases[packageName];
        const releaseResult = releases
            ? {
                releases,
                sourceUrl: 'https://github.com/actions/runner-images',
            }
            : null;
        return Promise.resolve(releaseResult);
    }
}
exports.GithubRunnersDatasource = GithubRunnersDatasource;
//# sourceMappingURL=index.js.map