"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GiteaTagsDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const gitea_1 = require("../../../util/http/gitea");
const regex_1 = require("../../../util/regex");
const url_1 = require("../../../util/url");
const datasource_1 = require("../datasource");
const schema_1 = require("./schema");
class GiteaTagsDatasource extends datasource_1.Datasource {
    static id = 'gitea-tags';
    http = new gitea_1.GiteaHttp(GiteaTagsDatasource.id);
    static defaultRegistryUrls = ['https://gitea.com'];
    static cacheNamespace = `datasource-${GiteaTagsDatasource.id}`;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `created` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    constructor() {
        super(GiteaTagsDatasource.id);
    }
    static getRegistryURL(registryUrl) {
        // fallback to default API endpoint if custom not provided
        return registryUrl ?? this.defaultRegistryUrls[0];
    }
    static getApiUrl(registryUrl) {
        const res = GiteaTagsDatasource.getRegistryURL(registryUrl).replace((0, regex_1.regEx)(/\/api\/v1$/), '');
        return `${(0, url_1.ensureTrailingSlash)(res)}api/v1/`;
    }
    static getCacheKey(registryUrl, repo, type) {
        return `${GiteaTagsDatasource.getRegistryURL(registryUrl)}:${repo}:${type}`;
    }
    static getSourceUrl(packageName, registryUrl) {
        const url = GiteaTagsDatasource.getRegistryURL(registryUrl);
        const normalizedUrl = (0, url_1.ensureTrailingSlash)(url);
        return `${normalizedUrl}${packageName}`;
    }
    // getReleases fetches list of tags for the repository
    async getReleases({ registryUrl, packageName: repo, }) {
        const url = `${GiteaTagsDatasource.getApiUrl(registryUrl)}repos/${repo}/tags`;
        const tags = (await this.http.getJson(url, {
            paginate: true,
        }, schema_1.TagsSchema)).body;
        const dependency = {
            sourceUrl: GiteaTagsDatasource.getSourceUrl(repo, registryUrl),
            registryUrl: GiteaTagsDatasource.getRegistryURL(registryUrl),
            releases: tags.map(({ name, commit }) => ({
                version: name,
                gitRef: name,
                newDigest: commit.sha,
                releaseTimestamp: commit.created,
            })),
        };
        return dependency;
    }
    // getTagCommit fetched the commit has for specified tag
    async getTagCommit(registryUrl, repo, tag) {
        const url = `${GiteaTagsDatasource.getApiUrl(registryUrl)}repos/${repo}/tags/${tag}`;
        const { body } = await this.http.getJson(url, schema_1.TagSchema);
        return body.commit.sha;
    }
    // getDigest fetched the latest commit for repository main branch
    // however, if newValue is provided, then getTagCommit is called
    async getDigest({ packageName: repo, registryUrl }, newValue) {
        if (newValue?.length) {
            return this.getTagCommit(registryUrl, repo, newValue);
        }
        const url = `${GiteaTagsDatasource.getApiUrl(registryUrl)}repos/${repo}/commits?stat=false&verification=false&files=false&page=1&limit=1`;
        const { body } = await this.http.getJson(url, schema_1.CommitsSchema);
        if (body.length === 0) {
            return null;
        }
        return body[0].sha;
    }
}
exports.GiteaTagsDatasource = GiteaTagsDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: GiteaTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => GiteaTagsDatasource.getCacheKey(registryUrl, packageName, 'tags'),
    })
], GiteaTagsDatasource.prototype, "getReleases", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: GiteaTagsDatasource.cacheNamespace,
        key: (registryUrl, repo, tag) => GiteaTagsDatasource.getCacheKey(registryUrl, repo, `tag-${tag}`),
    })
], GiteaTagsDatasource.prototype, "getTagCommit", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: GiteaTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => GiteaTagsDatasource.getCacheKey(registryUrl, packageName, 'digest'),
    })
], GiteaTagsDatasource.prototype, "getDigest", null);
//# sourceMappingURL=index.js.map