"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GalaxyDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const decorator_1 = require("../../../util/cache/package/decorator");
const pep440Versioning = tslib_1.__importStar(require("../../versioning/pep440"));
const datasource_1 = require("../datasource");
const schema_1 = require("./schema");
class GalaxyDatasource extends datasource_1.Datasource {
    static id = 'galaxy';
    constructor() {
        super(GalaxyDatasource.id);
    }
    customRegistrySupport = false;
    defaultRegistryUrls = ['https://galaxy.ansible.com/'];
    defaultVersioning = pep440Versioning.id;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `created` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined from the `github_user` and `github_repo` fields in the results.';
    async getReleases({ packageName, registryUrl, }) {
        const lookUp = packageName.split('.');
        const userName = lookUp[0];
        const projectName = lookUp[1];
        const galaxyAPIUrl = `${registryUrl}api/v1/roles/?owner__username=${userName}&name=${projectName}`;
        const galaxyProjectUrl = `${registryUrl}${userName}/${projectName}`;
        let body = null;
        try {
            const raw = await this.http.getJson(galaxyAPIUrl, schema_1.GalaxyV1);
            body = raw.body;
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        if (body.results.length > 1) {
            body.results = body.results.filter((result) => result.github_user === userName);
            if (!body.results.length) {
                logger_1.logger.warn({ dependency: packageName, userName }, `No matching result from galaxy for package`);
                return null;
            }
        }
        if (body.results.length === 0) {
            logger_1.logger.debug(`Received no results for ${packageName} from ${galaxyAPIUrl} `);
            return null;
        }
        const resultObject = body.results[0];
        const versions = resultObject.summary_fields.versions;
        const result = {
            releases: [],
        };
        result.dependencyUrl = galaxyProjectUrl;
        const { github_user: user, github_repo: repo } = resultObject;
        if (is_1.default.nonEmptyString(user) && is_1.default.nonEmptyString(repo)) {
            result.sourceUrl = `https://github.com/${user}/${repo}`;
        }
        result.releases = versions.map(({ version, releaseTimestamp }) => {
            const release = { version };
            if (releaseTimestamp) {
                release.releaseTimestamp = releaseTimestamp;
            }
            return release;
        });
        return result;
    }
}
exports.GalaxyDatasource = GalaxyDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: 'datasource-galaxy',
        key: (getReleasesConfig) => getReleasesConfig.packageName,
    })
], GalaxyDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map