"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlutterVersionDatasource = exports.stableVersionRegex = void 0;
const regex_1 = require("../../../util/regex");
const timestamp_1 = require("../../../util/timestamp");
const semver_1 = require("../../versioning/semver");
const datasource_1 = require("../datasource");
exports.stableVersionRegex = (0, regex_1.regEx)(/^\d+\.\d+\.\d+$/);
class FlutterVersionDatasource extends datasource_1.Datasource {
    static id = 'flutter-version';
    constructor() {
        super(FlutterVersionDatasource.id);
    }
    customRegistrySupport = false;
    defaultRegistryUrls = ['https://storage.googleapis.com'];
    caching = true;
    defaultVersioning = semver_1.id;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `release_date` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'We use the URL: https://github.com/flutter/flutter.';
    async getReleases({ registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const result = {
            homepage: 'https://flutter.dev',
            sourceUrl: 'https://github.com/flutter/flutter',
            registryUrl,
            releases: [],
        };
        try {
            const resp = (await this.http.getJsonUnchecked(`${registryUrl}/flutter_infra_release/releases/releases_linux.json`)).body;
            result.releases = resp.releases
                // The API response contains a stable version being released as a non-stable
                // release. And so we filter out these releases here.
                .filter(({ version, channel }) => {
                if (exports.stableVersionRegex.test(version)) {
                    return channel === 'stable';
                }
                return true;
            })
                .map(({ version, release_date, channel }) => ({
                version,
                releaseTimestamp: (0, timestamp_1.asTimestamp)(release_date),
                isStable: channel === 'stable',
            }));
            return result.releases.length ? result : null;
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
    }
}
exports.FlutterVersionDatasource = FlutterVersionDatasource;
//# sourceMappingURL=index.js.map