"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseChecksumsFromInRelease = parseChecksumsFromInRelease;
exports.computeFileChecksum = computeFileChecksum;
const fs_1 = require("../../../util/fs");
const hash_1 = require("../../../util/hash");
const regex_1 = require("../../../util/regex");
/**
 * Parses the SHA256 checksum for a specified package path from the InRelease content.
 *
 * @param inReleaseContent - content of the InRelease file
 * @param packagePath - path of the package file (e.g., 'contrib/binary-amd64/Packages.gz')
 * @returns The SHA256 checksum if found, otherwise undefined
 */
function parseChecksumsFromInRelease(inReleaseContent, packagePath) {
    const lines = inReleaseContent.split(regex_1.newlineRegex);
    const regex = (0, regex_1.regEx)(`([a-f0-9]{64})\\s+\\d+\\s+${(0, regex_1.escapeRegExp)(packagePath)}$`);
    for (const line of lines) {
        const match = regex.exec(line);
        if (match) {
            return match[1];
        }
    }
    return null;
}
/**
 * Computes the SHA256 checksum of a specified file.
 *
 * @param filePath - path of the file
 * @returns resolves to the SHA256 checksum
 */
function computeFileChecksum(filePath) {
    const stream = (0, fs_1.createCacheReadStream)(filePath);
    return (0, hash_1.hashStream)(stream, 'sha256');
}
//# sourceMappingURL=checksum.js.map