"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CpanDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const url_1 = require("../../../util/url");
const perlVersioning = tslib_1.__importStar(require("../../versioning/perl"));
const datasource_1 = require("../datasource");
const schema_1 = require("./schema");
class CpanDatasource extends datasource_1.Datasource {
    static id = 'cpan';
    constructor() {
        super(CpanDatasource.id);
    }
    customRegistrySupport = false;
    defaultRegistryUrls = ['https://fastapi.metacpan.org/'];
    defaultVersioning = perlVersioning.id;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `date` field in the results.';
    async getReleases({ packageName, registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        let result = null;
        const searchUrl = (0, url_1.joinUrlParts)(registryUrl, 'v1/file/_search');
        let releases = null;
        try {
            const body = {
                query: {
                    filtered: {
                        query: { match_all: {} },
                        filter: {
                            and: [
                                { term: { 'module.name': packageName } },
                                { term: { 'module.authorized': true } },
                                { exists: { field: 'module.associated_pod' } },
                            ],
                        },
                    },
                },
                _source: [
                    'module.name',
                    'module.version',
                    'distribution',
                    'date',
                    'deprecated',
                    'maturity',
                    'status',
                ],
                sort: [{ date: 'desc' }],
            };
            releases = (await this.http.postJson(searchUrl, { body }, schema_1.MetaCpanApiFileSearchResponse)).body;
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        let latestDistribution = null;
        let latestVersion = null;
        if (releases) {
            for (const release of releases) {
                latestDistribution ??= release.distribution;
                if (!latestVersion && release.isLatest) {
                    latestVersion = release.version;
                }
            }
        }
        if (releases.length > 0 && latestDistribution) {
            result = {
                releases,
                changelogUrl: `https://metacpan.org/dist/${latestDistribution}/changes`,
                homepage: `https://metacpan.org/pod/${packageName}`,
            };
            if (latestVersion) {
                result.tags ??= {};
                result.tags.latest = latestVersion;
            }
        }
        return result;
    }
}
exports.CpanDatasource = CpanDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${CpanDatasource.id}`,
        key: ({ packageName }) => `${packageName}`,
    })
], CpanDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map