"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getReleases = getReleases;
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const timestamp_1 = require("../../../util/timestamp");
const prefix_dev_1 = require("./schema/prefix-dev");
const MAX_PREFIX_DEV_GRAPHQL_PAGE = 100;
const query = `
query search($channel: String!, $package: String!, $page: Int = 0) {
  package(channelName: $channel, name: $package) {
    variants(limit: 500, page: $page) {
      pages
      page {
        createdAt
        version
        yankedReason
        urls {
          url
          kind
        }
      }
    }
  }
}
`;
async function getReleases(http, channel, packageName) {
    logger_1.logger.debug({ channel, packageName }, 'lookup package from prefix.dev graphql API');
    const files = await getPagedResponse(http, query, {
        channel,
        package: packageName,
    });
    if (!files.length) {
        return null;
    }
    let homepage = undefined;
    let sourceUrl = undefined;
    const releases = {};
    for (const file of files) {
        const version = file.version;
        homepage ??= file.urls.HOME;
        sourceUrl ??= file.urls.DEV;
        releases[version] ??= { version };
        // we assume all packages are roughly released on the same time
        releases[version].releaseTimestamp =
            releases[version].releaseTimestamp ??
                timestamp_1.MaybeTimestamp.parse(file.createdAt);
        // if the version has not been marked as deprecated, check other releases packages of the same version
        releases[version].isDeprecated ??= (0, array_1.isNotNullOrUndefined)(file.yankedReason);
    }
    return {
        homepage,
        sourceUrl,
        releases: Object.values(releases),
    };
}
async function getPagedResponse(http, query, data) {
    const result = [];
    for (let page = 0; page <= MAX_PREFIX_DEV_GRAPHQL_PAGE; page++) {
        const res = await http.postJson('https://prefix.dev/api/graphql', {
            body: {
                operationName: 'search',
                query,
                variables: {
                    ...data,
                    page,
                },
            },
        }, prefix_dev_1.PagedResponseSchema);
        const currentPage = res.body.data.package?.variants;
        if (!currentPage) {
            break;
        }
        result.push(...currentPage.page);
        if (page >= currentPage.pages - 1) {
            break;
        }
    }
    return result;
}
//# sourceMappingURL=prefix-dev.js.map