"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdnjsDatasource = void 0;
const tslib_1 = require("tslib");
const zod_1 = require("zod");
const logger_1 = require("../../../logger");
const external_host_error_1 = require("../../../types/errors/external-host-error");
const decorator_1 = require("../../../util/cache/package/decorator");
const memory_http_cache_provider_1 = require("../../../util/http/cache/memory-http-cache-provider");
const result_1 = require("../../../util/result");
const datasource_1 = require("../datasource");
const schema_1 = require("../schema");
const schema_2 = require("./schema");
class CdnjsDatasource extends datasource_1.Datasource {
    static id = 'cdnjs';
    constructor() {
        super(CdnjsDatasource.id);
    }
    customRegistrySupport = false;
    defaultRegistryUrls = ['https://api.cdnjs.com/'];
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined from the `repository` field in the results.';
    async getReleases(config) {
        const result = result_1.Result.parse(config, schema_1.ReleasesConfig)
            .transform(({ packageName, registryUrl }) => {
            const [library] = packageName.split('/');
            const url = `${registryUrl}libraries/${library}?fields=homepage,repository,versions`;
            return this.http.getJsonSafe(url, { cacheProvider: memory_http_cache_provider_1.memCacheProvider }, schema_2.CdnjsAPIVersionResponseSchema);
        })
            .transform(({ versions, homepage, repository }) => {
            const releases = versions;
            const res = { releases };
            if (homepage) {
                res.homepage = homepage;
            }
            if (repository) {
                res.sourceUrl = repository;
            }
            return res;
        });
        const { val, err } = await result.unwrap();
        if (err instanceof zod_1.ZodError) {
            logger_1.logger.debug({ err }, 'cdnjs: validation error');
            return null;
        }
        if (err) {
            this.handleGenericErrors(err);
        }
        return val;
    }
    async getDigest(config, newValue) {
        const { packageName } = config;
        const [library] = packageName.split('/');
        const assetName = packageName.replace(`${library}/`, '');
        const result = result_1.Result.parse(config, schema_1.DigestsConfig)
            .transform(({ registryUrl }) => {
            const url = `${registryUrl}libraries/${library}/${newValue}?fields=sri`;
            return this.http.getJsonSafe(url, schema_2.CdnjsAPISriResponseSchema);
        })
            .transform(({ sri }) => {
            return sri?.[assetName];
        });
        const { val = null, err } = await result.unwrap();
        if (err instanceof zod_1.ZodError) {
            logger_1.logger.debug({ err }, 'cdnjs: validation error');
            return null;
        }
        if (err) {
            this.handleGenericErrors(err);
        }
        return val;
    }
    handleHttpErrors(err) {
        if (err.response?.statusCode !== 404) {
            throw new external_host_error_1.ExternalHostError(err);
        }
    }
}
exports.CdnjsDatasource = CdnjsDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${CdnjsDatasource.id}`,
        key: ({ packageName }) => {
            const library = packageName.split('/')[0];
            return `getReleases:${library}`;
        },
    })
], CdnjsDatasource.prototype, "getReleases", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${CdnjsDatasource.id}`,
        key: ({ registryUrl, packageName }, newValue) => `getDigest:${registryUrl}:${packageName}:${newValue}}`,
    })
], CdnjsDatasource.prototype, "getDigest", null);
//# sourceMappingURL=index.js.map