"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BitriseDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const decorator_1 = require("../../../util/cache/package/decorator");
const common_1 = require("../../../util/common");
const url_1 = require("../../../util/git/url");
const github_1 = require("../../../util/http/github");
const string_1 = require("../../../util/string");
const url_2 = require("../../../util/url");
const schema_1 = require("../../platform/github/schema");
const semver_1 = tslib_1.__importDefault(require("../../versioning/semver"));
const datasource_1 = require("../datasource");
const schema_2 = require("./schema");
class BitriseDatasource extends datasource_1.Datasource {
    static id = 'bitrise';
    http;
    constructor() {
        super(BitriseDatasource.id);
        this.http = new github_1.GithubHttp(this.id);
    }
    customRegistrySupport = true;
    defaultRegistryUrls = [
        'https://github.com/bitrise-io/bitrise-steplib.git',
    ];
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `published_at` field in the results.';
    sourceUrlSupport = 'release';
    sourceUrlNote = 'The source URL is determined from the `source_code_url` field of the release object in the results.';
    async getReleases({ packageName, registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const parsedUrl = (0, url_1.parseGitUrl)(registryUrl);
        if ((0, common_1.detectPlatform)(registryUrl) !== 'github') {
            logger_1.logger.once.warn(`${parsedUrl.source} is not a supported Git hoster for this datasource`);
            return null;
        }
        const result = {
            releases: [],
        };
        const massagedPackageName = encodeURIComponent(packageName);
        const baseApiURL = parsedUrl.resource === 'github.com'
            ? 'https://api.github.com'
            : `https://${parsedUrl.resource}/api/v3`;
        const packageUrl = (0, url_2.joinUrlParts)(baseApiURL, 'repos', parsedUrl.full_name, 'contents/steps', massagedPackageName);
        const { body: packageRaw } = await this.http.getJson(packageUrl, schema_1.GithubContentResponse);
        if (!is_1.default.array(packageRaw)) {
            logger_1.logger.warn({ data: packageRaw, url: packageUrl }, 'Got unexpected response for Bitrise package location');
            return null;
        }
        for (const versionDir of packageRaw.filter((element) => semver_1.default.isValid(element.name))) {
            const stepUrl = (0, url_2.joinUrlParts)(packageUrl, versionDir.name, 'step.yml');
            // TODO use getRawFile when ready #30155
            const { body } = await this.http.getJson(stepUrl, schema_1.GithubContentResponse);
            if (!('content' in body)) {
                logger_1.logger.warn({ data: body, url: stepUrl }, 'Got unexpected response for Bitrise step location');
                return null;
            }
            if (body.encoding !== 'base64') {
                logger_1.logger.warn({ encoding: body.encoding, data: body, url: stepUrl }, `Got unexpected encoding for Bitrise step location`);
                return null;
            }
            const content = (0, string_1.fromBase64)(body.content);
            const { published_at, source_code_url } = schema_2.BitriseStepFile.parse(content);
            result.releases.push({
                version: versionDir.name,
                releaseTimestamp: published_at,
                sourceUrl: source_code_url,
            });
        }
        // if we have no releases return null
        if (!result.releases.length) {
            return null;
        }
        return {
            ...result,
            homepage: `https://bitrise.io/integrations/steps/${packageName}`,
        };
    }
}
exports.BitriseDatasource = BitriseDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${BitriseDatasource.id}`,
        key: ({ packageName, registryUrl }) => `${registryUrl}/${packageName}`,
    })
], BitriseDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map