"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BitbucketTagsDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const bitbucket_1 = require("../../../util/http/bitbucket");
const timestamp_1 = require("../../../util/timestamp");
const url_1 = require("../../../util/url");
const schema_1 = require("../../platform/bitbucket/schema");
const datasource_1 = require("../datasource");
class BitbucketTagsDatasource extends datasource_1.Datasource {
    static id = 'bitbucket-tags';
    bitbucketHttp = new bitbucket_1.BitbucketHttp(BitbucketTagsDatasource.id);
    static customRegistrySupport = true;
    static registryStrategy = 'first';
    static defaultRegistryUrls = ['https://bitbucket.org'];
    static releaseTimestampSupport = true;
    static releaseTimestampNote = 'The release timestamp is determined from the `date` field in the results.';
    static sourceUrlSupport = 'package';
    static sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    static cacheNamespace = `datasource-${BitbucketTagsDatasource.id}`;
    constructor() {
        super(BitbucketTagsDatasource.id);
    }
    static getRegistryURL(registryUrl) {
        // fallback to default API endpoint if custom not provided
        return registryUrl ?? this.defaultRegistryUrls[0];
    }
    static getCacheKey(registryUrl, repo, type) {
        return `${BitbucketTagsDatasource.getRegistryURL(registryUrl)}:${repo}:${type}`;
    }
    static getSourceUrl(packageName, registryUrl) {
        const url = BitbucketTagsDatasource.getRegistryURL(registryUrl);
        const normalizedUrl = (0, url_1.ensureTrailingSlash)(url);
        return `${normalizedUrl}${packageName}`;
    }
    // getReleases fetches list of tags for the repository
    async getReleases({ registryUrl, packageName: repo, }) {
        const url = `/2.0/repositories/${repo}/refs/tags`;
        const bitbucketTags = (await this.bitbucketHttp.getJsonUnchecked(url, {
            paginate: true,
        })).body.values;
        const dependency = {
            sourceUrl: BitbucketTagsDatasource.getSourceUrl(repo, registryUrl),
            registryUrl: BitbucketTagsDatasource.getRegistryURL(registryUrl),
            releases: bitbucketTags.map(({ name, target }) => ({
                version: name,
                gitRef: name,
                releaseTimestamp: (0, timestamp_1.asTimestamp)(target?.date),
            })),
        };
        return dependency;
    }
    // getTagCommit fetched the commit has for specified tag
    async getTagCommit(_registryUrl, repo, tag) {
        const url = `/2.0/repositories/${repo}/refs/tags/${tag}`;
        const bitbucketTag = (await this.bitbucketHttp.getJsonUnchecked(url)).body;
        return bitbucketTag.target?.hash ?? null;
    }
    async getMainBranch(_registryUrl, repo) {
        return (await this.bitbucketHttp.getJson(`/2.0/repositories/${repo}`, schema_1.RepoInfo)).body.mainbranch;
    }
    // getDigest fetched the latest commit for repository main branch
    // however, if newValue is provided, then getTagCommit is called
    async getDigest({ packageName: repo, registryUrl }, newValue) {
        if (newValue?.length) {
            return this.getTagCommit(registryUrl, repo, newValue);
        }
        const mainBranch = await this.getMainBranch(BitbucketTagsDatasource.getRegistryURL(registryUrl), repo);
        const url = `/2.0/repositories/${repo}/commits/${mainBranch}`;
        const bitbucketCommits = (await this.bitbucketHttp.getJsonUnchecked(url)).body;
        if (bitbucketCommits.values.length === 0) {
            return null;
        }
        return bitbucketCommits.values[0].hash;
    }
}
exports.BitbucketTagsDatasource = BitbucketTagsDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => BitbucketTagsDatasource.getCacheKey(registryUrl, packageName, 'tags'),
    })
], BitbucketTagsDatasource.prototype, "getReleases", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketTagsDatasource.cacheNamespace,
        key: (registryUrl, repo, tag) => BitbucketTagsDatasource.getCacheKey(registryUrl, repo, `tag-${tag}`),
    })
], BitbucketTagsDatasource.prototype, "getTagCommit", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketTagsDatasource.cacheNamespace,
        key: (registryUrl, repo) => BitbucketTagsDatasource.getCacheKey(registryUrl, repo, 'mainbranch'),
        ttlMinutes: 60,
    })
], BitbucketTagsDatasource.prototype, "getMainBranch", null);
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: BitbucketTagsDatasource.cacheNamespace,
        key: ({ registryUrl, packageName }) => BitbucketTagsDatasource.getCacheKey(registryUrl, packageName, 'digest'),
    })
], BitbucketTagsDatasource.prototype, "getDigest", null);
//# sourceMappingURL=index.js.map