"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AzurePipelinesTasksDatasource = void 0;
const tslib_1 = require("tslib");
const global_1 = require("../../../config/global");
const decorator_1 = require("../../../util/cache/package/decorator");
const hostRules = tslib_1.__importStar(require("../../../util/host-rules"));
const loose_1 = require("../../versioning/loose");
const datasource_1 = require("../datasource");
const schema_1 = require("./schema");
const TASKS_URL_BASE = 'https://raw.githubusercontent.com/renovatebot/azure-devops-marketplace/main';
const BUILT_IN_TASKS_URL = `${TASKS_URL_BASE}/azure-pipelines-builtin-tasks.json`;
const MARKETPLACE_TASKS_URL = `${TASKS_URL_BASE}/azure-pipelines-marketplace-tasks.json`;
const BUILT_IN_TASKS_CHANGELOG_URL = 'https://github.com/microsoft/azure-pipelines-tasks/releases';
class AzurePipelinesTasksDatasource extends datasource_1.Datasource {
    static id = 'azure-pipelines-tasks';
    constructor() {
        super(AzurePipelinesTasksDatasource.id);
    }
    customRegistrySupport = false;
    defaultVersioning = loose_1.id;
    async getReleases({ packageName, }) {
        const platform = global_1.GlobalConfig.get('platform');
        const endpoint = global_1.GlobalConfig.get('endpoint');
        const { token } = hostRules.find({
            hostType: AzurePipelinesTasksDatasource.id,
            url: endpoint,
        });
        if (platform === 'azure' && endpoint && token) {
            const auth = Buffer.from(`renovate:${token}`).toString('base64');
            const opts = {
                headers: { authorization: `Basic ${auth}` },
            };
            const results = await this.getTasks(`${endpoint}/_apis/distributedtask/tasks/`, opts, schema_1.AzurePipelinesJSON);
            const result = { releases: [] };
            results.value
                .filter((task) => {
                const matchers = [
                    task.id === packageName,
                    task.name === packageName,
                    task.contributionIdentifier !== null &&
                        `${task.contributionIdentifier}.${task.id}` === packageName,
                    task.contributionIdentifier !== null &&
                        `${task.contributionIdentifier}.${task.name}` === packageName,
                ];
                return matchers.some((match) => match);
            })
                .sort(AzurePipelinesTasksDatasource.compareSemanticVersions('version'))
                .forEach((task) => {
                const release = {
                    version: `${task.version.major}.${task.version.minor}.${task.version.patch}`,
                    changelogContent: task.releaseNotes,
                    isDeprecated: task.deprecated,
                };
                if (task.serverOwned) {
                    release.changelogUrl = BUILT_IN_TASKS_CHANGELOG_URL;
                }
                result.releases.push(release);
            });
            return result;
        }
        else {
            const versions = (await this.getTasks(BUILT_IN_TASKS_URL, {}, schema_1.AzurePipelinesFallbackTasks))[packageName.toLowerCase()] ??
                (await this.getTasks(MARKETPLACE_TASKS_URL, {}, schema_1.AzurePipelinesFallbackTasks))[packageName.toLowerCase()];
            if (versions) {
                const releases = versions.map((version) => ({ version }));
                return { releases };
            }
        }
        return null;
    }
    async getTasks(url, opts, schema) {
        const { body } = await this.http.getJson(url, opts, schema);
        return body;
    }
    static compareSemanticVersions = (key) => (a, b) => {
        const a1Version = schema_1.AzurePipelinesTaskVersion.safeParse(a[key]).data;
        const b1Version = schema_1.AzurePipelinesTaskVersion.safeParse(b[key]).data;
        const a1 = a1Version === undefined
            ? ''
            : `${a1Version.major}.${a1Version.minor}.${a1Version.patch}`;
        const b1 = b1Version === undefined
            ? ''
            : `${b1Version.major}.${b1Version.minor}.${b1Version.patch}`;
        const len = Math.min(a1.length, b1.length);
        for (let i = 0; i < len; i++) {
            const a2 = +a1[i] || 0;
            const b2 = +b1[i] || 0;
            if (a2 !== b2) {
                return a2 > b2 ? 1 : -1;
            }
        }
        return b1.length - a1.length;
    };
}
exports.AzurePipelinesTasksDatasource = AzurePipelinesTasksDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${AzurePipelinesTasksDatasource.id}`,
        key: (url) => url,
        ttlMinutes: 24 * 60,
    })
], AzurePipelinesTasksDatasource.prototype, "getTasks", null);
//# sourceMappingURL=index.js.map