"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenovateLogger = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const once_1 = require("./once");
const remap_1 = require("./remap");
const utils_1 = require("./utils");
const loggerLevels = [
    'trace',
    'debug',
    'info',
    'warn',
    'error',
    'fatal',
];
class RenovateLogger {
    bunyanLogger;
    context;
    meta;
    logger = { once: { reset: once_1.reset } };
    once = this.logger.once;
    constructor(bunyanLogger, context, meta) {
        this.bunyanLogger = bunyanLogger;
        this.context = context;
        this.meta = meta;
        for (const level of loggerLevels) {
            this.logger[level] = this.logFactory(level);
            this.logger.once[level] = this.logOnceFn(level);
        }
    }
    trace(p1, p2) {
        this.log('trace', p1, p2);
    }
    debug(p1, p2) {
        this.log('debug', p1, p2);
    }
    info(p1, p2) {
        this.log('info', p1, p2);
    }
    warn(p1, p2) {
        this.log('warn', p1, p2);
    }
    error(p1, p2) {
        this.log('error', p1, p2);
    }
    fatal(p1, p2) {
        this.log('fatal', p1, p2);
    }
    addStream(stream) {
        this.bunyanLogger.addStream((0, utils_1.withSanitizer)(stream));
    }
    childLogger() {
        return new RenovateLogger(this.bunyanLogger.child({}), this.context, this.meta);
    }
    get logContext() {
        return this.context;
    }
    set logContext(context) {
        this.context = context;
    }
    setMeta(obj) {
        this.meta = { ...obj };
    }
    addMeta(obj) {
        this.meta = { ...this.meta, ...obj };
    }
    removeMeta(fields) {
        for (const key of Object.keys(this.meta)) {
            if (fields.includes(key)) {
                delete this.meta[key];
            }
        }
    }
    logFactory(_level) {
        return (p1, p2) => {
            const meta = {
                logContext: this.context,
                ...this.meta,
                ...(0, utils_1.toMeta)(p1),
            };
            const msg = (0, utils_1.getMessage)(p1, p2);
            let level = _level;
            if (is_1.default.string(msg)) {
                const remappedLevel = (0, remap_1.getRemappedLevel)(msg);
                /* v8 ignore next 4 -- not easily testable */
                if (remappedLevel) {
                    meta.oldLevel = level;
                    level = remappedLevel;
                }
                this.bunyanLogger[level](meta, msg);
            }
            else {
                this.bunyanLogger[level](meta);
            }
        };
    }
    logOnceFn(level) {
        const logOnceFn = (p1, p2) => {
            (0, once_1.once)(() => {
                const logFn = this[level].bind(this); // bind to the instance.
                if (is_1.default.string(p1)) {
                    logFn(p1);
                }
                else {
                    logFn(p1, p2);
                }
            }, logOnceFn);
        };
        return logOnceFn;
    }
    log(level, p1, p2) {
        const logFn = this.logger[level];
        if (is_1.default.string(p1)) {
            logFn(p1);
        }
        else {
            logFn(p1, p2);
        }
    }
}
exports.RenovateLogger = RenovateLogger;
//# sourceMappingURL=renovate-logger.js.map