"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.logger = void 0;
exports.logLevel = logLevel;
exports.createDefaultStreams = createDefaultStreams;
exports.setContext = setContext;
exports.getContext = getContext;
exports.setMeta = setMeta;
exports.addMeta = addMeta;
exports.removeMeta = removeMeta;
exports.withMeta = withMeta;
exports.addStream = addStream;
exports.levels = levels;
exports.getProblems = getProblems;
exports.clearProblems = clearProblems;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const bunyan = tslib_1.__importStar(require("bunyan"));
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const nanoid_1 = require("nanoid");
const upath_1 = tslib_1.__importDefault(require("upath"));
const cmd_serializer_1 = tslib_1.__importDefault(require("./cmd-serializer"));
const config_serializer_1 = tslib_1.__importDefault(require("./config-serializer"));
const err_serializer_1 = tslib_1.__importDefault(require("./err-serializer"));
const pretty_stdout_1 = require("./pretty-stdout");
const renovate_logger_1 = require("./renovate-logger");
const utils_1 = require("./utils");
const problems = new utils_1.ProblemStream();
let stdoutLevel = (0, utils_1.validateLogLevel)((0, utils_1.getEnv)('LOG_LEVEL'), 'info');
function logLevel() {
    return stdoutLevel;
}
function createDefaultStreams(stdoutLevel, problems, logFile) {
    const stdout = {
        name: 'stdout',
        level: stdoutLevel,
        stream: process.stdout,
    };
    if ((0, utils_1.getEnv)('LOG_FORMAT') !== 'json') {
        // TODO: typings (#9615)
        const prettyStdOut = new pretty_stdout_1.RenovateStream();
        prettyStdOut.pipe(process.stdout);
        stdout.stream = prettyStdOut;
        stdout.type = 'raw';
    }
    const problemsStream = {
        name: 'problems',
        level: 'warn',
        stream: problems,
        type: 'raw',
    };
    const logFileStream = is_1.default.string(logFile)
        ? createLogFileStream(logFile)
        : undefined;
    return [stdout, problemsStream, logFileStream].filter(Boolean);
}
function createLogFileStream(logFile) {
    // Ensure log file directory exists
    const directoryName = upath_1.default.dirname(logFile);
    fs_extra_1.default.ensureDirSync(directoryName);
    return {
        name: 'logfile',
        path: logFile,
        level: (0, utils_1.validateLogLevel)((0, utils_1.getEnv)('LOG_FILE_LEVEL'), 'debug'),
    };
}
function serializedSanitizedLogger(streams) {
    return bunyan.createLogger({
        name: 'renovate',
        serializers: {
            body: config_serializer_1.default,
            cmd: cmd_serializer_1.default,
            config: config_serializer_1.default,
            migratedConfig: config_serializer_1.default,
            originalConfig: config_serializer_1.default,
            presetConfig: config_serializer_1.default,
            oldConfig: config_serializer_1.default,
            newConfig: config_serializer_1.default,
            err: err_serializer_1.default,
        },
        streams: streams.map(utils_1.withSanitizer),
    });
}
const defaultStreams = createDefaultStreams(stdoutLevel, problems, (0, utils_1.getEnv)('LOG_FILE'));
const bunyanLogger = serializedSanitizedLogger(defaultStreams);
const logContext = (0, utils_1.getEnv)('LOG_CONTEXT') ?? (0, nanoid_1.nanoid)();
const loggerInternal = new renovate_logger_1.RenovateLogger(bunyanLogger, logContext, {});
exports.logger = loggerInternal;
function setContext(value) {
    loggerInternal.logContext = value;
}
function getContext() {
    return loggerInternal.logContext;
}
// setMeta overrides existing meta, may remove fields if no longer existing
function setMeta(obj) {
    loggerInternal.setMeta(obj);
}
// addMeta overrides or adds fields but does not remove any
function addMeta(obj) {
    loggerInternal.addMeta(obj);
}
// removeMeta removes the provided fields from meta
function removeMeta(fields) {
    loggerInternal.removeMeta(fields);
}
function withMeta(obj, cb) {
    addMeta(obj);
    try {
        return cb();
    }
    finally {
        removeMeta(Object.keys(obj));
    }
}
function addStream(stream) {
    loggerInternal.addStream(stream);
}
/**
 * For testing purposes only
 * @param name stream name
 * @param level log level
 * @private
 */
function levels(name, level) {
    bunyanLogger.levels(name, level);
    if (name === 'stdout') {
        stdoutLevel = level;
    }
}
function getProblems() {
    return problems.getProblems();
}
function clearProblems() {
    return problems.clearProblems();
}
//# sourceMappingURL=index.js.map