"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.init = init;
exports.shutdown = shutdown;
exports.disableInstrumentations = disableInstrumentations;
exports.getTracerProvider = getTracerProvider;
exports.instrument = instrument;
const tslib_1 = require("tslib");
const node_http_1 = require("node:http");
const api = tslib_1.__importStar(require("@opentelemetry/api"));
const api_1 = require("@opentelemetry/api");
const context_async_hooks_1 = require("@opentelemetry/context-async-hooks");
const exporter_trace_otlp_http_1 = require("@opentelemetry/exporter-trace-otlp-http");
const instrumentation_1 = require("@opentelemetry/instrumentation");
const instrumentation_bunyan_1 = require("@opentelemetry/instrumentation-bunyan");
const instrumentation_http_1 = require("@opentelemetry/instrumentation-http");
const resource_detector_aws_1 = require("@opentelemetry/resource-detector-aws");
const resource_detector_azure_1 = require("@opentelemetry/resource-detector-azure");
const resource_detector_gcp_1 = require("@opentelemetry/resource-detector-gcp");
const resource_detector_github_1 = require("@opentelemetry/resource-detector-github");
const resources_1 = require("@opentelemetry/resources");
const sdk_trace_base_1 = require("@opentelemetry/sdk-trace-base");
const sdk_trace_node_1 = require("@opentelemetry/sdk-trace-node");
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const expose_cjs_1 = require("../expose.cjs");
const env_1 = require("../util/env");
const utils_1 = require("./utils");
let instrumentations = [];
init();
function init() {
    if (!(0, utils_1.isTracingEnabled)()) {
        return;
    }
    const spanProcessors = [];
    // add processors
    if ((0, utils_1.isTraceDebuggingEnabled)()) {
        spanProcessors.push(new sdk_trace_base_1.SimpleSpanProcessor(new sdk_trace_base_1.ConsoleSpanExporter()));
    }
    // OTEL specification environment variable
    if ((0, utils_1.isTraceSendingEnabled)()) {
        const exporter = new exporter_trace_otlp_http_1.OTLPTraceExporter();
        spanProcessors.push(new sdk_trace_base_1.BatchSpanProcessor(exporter));
    }
    const env = (0, env_1.getEnv)();
    const baseResource = (0, resources_1.resourceFromAttributes)({
        // https://github.com/open-telemetry/opentelemetry-specification/blob/main/specification/resource/semantic_conventions/README.md#semantic-attributes-with-sdk-provided-default-value
        [semantic_conventions_1.ATTR_SERVICE_NAME]: env.OTEL_SERVICE_NAME ?? 'renovate',
        // https://github.com/open-telemetry/opentelemetry-js/tree/main/semantic-conventions#unstable-semconv
        // https://github.com/open-telemetry/opentelemetry-js/blob/e9d3c71918635d490b6a9ac9f8259265b38394d0/semantic-conventions/src/experimental_attributes.ts#L7688
        ['service.namespace']: env.OTEL_SERVICE_NAMESPACE ?? 'renovatebot.com',
        [semantic_conventions_1.ATTR_SERVICE_VERSION]: env.OTEL_SERVICE_VERSION ?? expose_cjs_1.pkg.version,
    });
    const detectedResource = (0, resources_1.detectResources)({
        detectors: [
            resource_detector_aws_1.awsBeanstalkDetector,
            resource_detector_aws_1.awsEc2Detector,
            resource_detector_aws_1.awsEcsDetector,
            resource_detector_aws_1.awsEksDetector,
            resource_detector_aws_1.awsLambdaDetector,
            resource_detector_azure_1.azureAppServiceDetector,
            resource_detector_azure_1.azureFunctionsDetector,
            resource_detector_azure_1.azureVmDetector,
            resource_detector_gcp_1.gcpDetector,
            resource_detector_github_1.gitHubDetector,
            resources_1.envDetector,
        ],
    });
    const traceProvider = new sdk_trace_node_1.NodeTracerProvider({
        resource: baseResource.merge(detectedResource),
        spanProcessors,
    });
    const contextManager = new context_async_hooks_1.AsyncLocalStorageContextManager();
    traceProvider.register({
        contextManager,
    });
    instrumentations = [
        new instrumentation_http_1.HttpInstrumentation({
            /* v8 ignore start -- not easily testable */
            applyCustomAttributesOnSpan: (span, request, response) => {
                // ignore 404 errors when the branch protection of Github could not be found. This is expected if no rules are configured
                if (request instanceof node_http_1.ClientRequest &&
                    request.host === `api.github.com` &&
                    request.path.endsWith(`/protection`) &&
                    response.statusCode === 404) {
                    span.setStatus({ code: api_1.SpanStatusCode.OK });
                }
            },
            /* v8 ignore stop */
        }),
        new instrumentation_bunyan_1.BunyanInstrumentation(),
    ];
    (0, instrumentation_1.registerInstrumentations)({
        instrumentations,
    });
}
/* v8 ignore start -- not easily testable */
// https://github.com/open-telemetry/opentelemetry-js-api/issues/34
async function shutdown() {
    const traceProvider = getTracerProvider();
    if (traceProvider instanceof sdk_trace_node_1.NodeTracerProvider) {
        await traceProvider.shutdown();
    }
    else if (traceProvider instanceof api_1.ProxyTracerProvider) {
        const delegateProvider = traceProvider.getDelegate();
        if (delegateProvider instanceof sdk_trace_node_1.NodeTracerProvider) {
            await delegateProvider.shutdown();
        }
    }
}
/* v8 ignore stop */
function disableInstrumentations() {
    for (const instrumentation of instrumentations) {
        instrumentation.disable();
    }
}
function getTracerProvider() {
    return api.trace.getTracerProvider();
}
function getTracer() {
    return getTracerProvider().getTracer('renovate');
}
function instrument(name, fn, options = {}, context = api.context.active()) {
    return getTracer().startActiveSpan(name, options, context, (span) => {
        try {
            const ret = fn();
            if (ret instanceof Promise) {
                return ret
                    .catch((e) => {
                    span.recordException(e);
                    span.setStatus({
                        code: api_1.SpanStatusCode.ERROR,
                        message: (0, utils_1.massageThrowable)(e),
                    });
                    throw e;
                })
                    .finally(() => span.end());
            }
            span.end();
            return ret;
        }
        catch (e) {
            span.recordException(e);
            span.setStatus({
                code: api_1.SpanStatusCode.ERROR,
                message: (0, utils_1.massageThrowable)(e),
            });
            span.end();
            throw e;
        }
    });
}
//# sourceMappingURL=index.js.map