"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseFileConfig = parseFileConfig;
const tslib_1 = require("tslib");
const json_dup_key_validator_1 = tslib_1.__importDefault(require("json-dup-key-validator"));
const json5_1 = tslib_1.__importDefault(require("json5"));
const strip_json_comments_1 = tslib_1.__importDefault(require("strip-json-comments"));
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../logger");
const common_1 = require("../util/common");
function parseFileConfig(fileName, fileContents) {
    const fileType = upath_1.default.extname(fileName);
    if (fileType === '.json5') {
        try {
            return { success: true, parsedContents: json5_1.default.parse(fileContents) };
        }
        catch (err) {
            logger_1.logger.debug({ fileName, fileContents }, 'Error parsing JSON5 file');
            const validationError = 'Invalid JSON5 (parsing failed)';
            const validationMessage = `JSON5.parse error: \`${err.message.replaceAll('`', "'")}\``;
            return {
                success: false,
                validationError,
                validationMessage,
            };
        }
    }
    else {
        const jsonString = (0, strip_json_comments_1.default)(fileContents);
        let allowDuplicateKeys = true;
        let jsonValidationError = json_dup_key_validator_1.default.validate(jsonString, allowDuplicateKeys);
        if (jsonValidationError) {
            const validationError = 'Invalid JSON (parsing failed)';
            const validationMessage = jsonValidationError;
            return {
                success: false,
                validationError,
                validationMessage,
            };
        }
        allowDuplicateKeys = false;
        jsonValidationError = json_dup_key_validator_1.default.validate(jsonString, allowDuplicateKeys);
        if (jsonValidationError) {
            const validationError = 'Duplicate keys in JSON';
            const validationMessage = JSON.stringify(jsonValidationError);
            return {
                success: false,
                validationError,
                validationMessage,
            };
        }
        try {
            return {
                success: true,
                parsedContents: (0, common_1.parseJson)(fileContents, fileName),
            };
        }
        catch (err) {
            logger_1.logger.debug({ fileContents }, 'Error parsing renovate config');
            const validationError = 'Invalid JSON (parsing failed)';
            const validationMessage = `JSON.parse error:  \`${err.message.replaceAll('`', "'")}\``;
            return { success: false, validationError, validationMessage };
        }
    }
}
//# sourceMappingURL=parse.js.map