"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fixShortHours = fixShortHours;
exports.migrateConfig = migrateConfig;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const dequal_1 = require("dequal");
const logger_1 = require("../logger");
const clone_1 = require("../util/clone");
const regex_1 = require("../util/regex");
const migrations_1 = require("./migrations");
const options_1 = require("./options");
const utils_1 = require("./utils");
const options = (0, options_1.getOptions)();
function fixShortHours(input) {
    return input.replace((0, regex_1.regEx)(/( \d?\d)((a|p)m)/g), '$1:00$2');
}
let optionTypes;
// Returns a migrated config
function migrateConfig(config, parentKey) {
    try {
        if (!optionTypes) {
            optionTypes = {};
            options.forEach((option) => {
                optionTypes[option.name] = option.type;
            });
        }
        const newConfig = migrations_1.MigrationsService.run(config, parentKey);
        const migratedConfig = (0, clone_1.clone)(newConfig);
        for (const [key, val] of Object.entries(newConfig)) {
            if (is_1.default.string(val) && val.includes('{{baseDir}}')) {
                migratedConfig[key] = val.replace((0, regex_1.regEx)(/{{baseDir}}/g), '{{packageFileDir}}');
            }
            else if (is_1.default.string(val) && val.includes('{{lookupName}}')) {
                migratedConfig[key] = val.replace((0, regex_1.regEx)(/{{lookupName}}/g), '{{packageName}}');
            }
            else if (is_1.default.string(val) && val.includes('{{depNameShort}}')) {
                migratedConfig[key] = val.replace((0, regex_1.regEx)(/{{depNameShort}}/g), '{{depName}}');
            }
            else if (is_1.default.string(val) && val.startsWith('{{semanticPrefix}}')) {
                migratedConfig[key] = val.replace('{{semanticPrefix}}', '{{#if semanticCommitType}}{{semanticCommitType}}{{#if semanticCommitScope}}({{semanticCommitScope}}){{/if}}: {{/if}}');
            }
            else if (optionTypes[key] === 'object' && is_1.default.boolean(val)) {
                migratedConfig[key] = { enabled: val };
            }
            else if (optionTypes[key] === 'boolean') {
                if (val === 'true') {
                    migratedConfig[key] = true;
                }
                else if (val === 'false') {
                    migratedConfig[key] = false;
                }
            }
            else if (optionTypes[key] === 'string' &&
                is_1.default.array(val) &&
                val.length === 1) {
                migratedConfig[key] = String(val[0]);
            }
            else if (is_1.default.array(val)) {
                if (is_1.default.array(migratedConfig?.[key])) {
                    const newArray = [];
                    for (const item of migratedConfig[key]) {
                        if (is_1.default.object(item) && !is_1.default.array(item)) {
                            const arrMigrate = migrateConfig(item);
                            newArray.push(arrMigrate.migratedConfig);
                        }
                        else {
                            newArray.push(item);
                        }
                    }
                    migratedConfig[key] = newArray;
                }
            }
            else if (is_1.default.object(val)) {
                const subMigrate = migrateConfig(migratedConfig[key], key);
                if (subMigrate.isMigrated) {
                    migratedConfig[key] = subMigrate.migratedConfig;
                }
            }
            const migratedTemplates = {
                fromVersion: 'currentVersion',
                newValueMajor: 'newMajor',
                newValueMinor: 'newMinor',
                newVersionMajor: 'newMajor',
                newVersionMinor: 'newMinor',
                toVersion: 'newVersion',
            };
            if (is_1.default.string(migratedConfig[key])) {
                for (const [from, to] of Object.entries(migratedTemplates)) {
                    migratedConfig[key] = migratedConfig[key].replace((0, regex_1.regEx)(from, 'g'), to);
                }
            }
        }
        const languages = [
            'docker',
            'dotnet',
            'golang',
            'java',
            'js',
            'node',
            'php',
            'python',
            'ruby',
            'rust',
        ];
        for (const language of languages) {
            if (is_1.default.nonEmptyObject(migratedConfig[language])) {
                migratedConfig.packageRules ??= [];
                const currentContent = migratedConfig[language];
                const packageRule = {
                    matchCategories: [language],
                    ...currentContent,
                };
                migratedConfig.packageRules.unshift(packageRule);
                delete migratedConfig[language];
            }
        }
        // Migrate nested packageRules
        if (is_1.default.nonEmptyArray(migratedConfig.packageRules)) {
            const existingRules = migratedConfig.packageRules;
            migratedConfig.packageRules = [];
            for (const packageRule of existingRules) {
                if (is_1.default.array(packageRule.packageRules)) {
                    logger_1.logger.debug('Flattening nested packageRules');
                    // merge each subrule and add to the parent list
                    for (const subrule of packageRule.packageRules) {
                        // TODO: fix types #22198
                        const combinedRule = (0, utils_1.mergeChildConfig)(packageRule, subrule);
                        delete combinedRule.packageRules;
                        migratedConfig.packageRules.push(combinedRule);
                    }
                }
                else {
                    migratedConfig.packageRules.push(packageRule);
                }
            }
        }
        if (is_1.default.nonEmptyObject(migratedConfig['pip-compile']) &&
            is_1.default.nonEmptyArray(migratedConfig['pip-compile'].managerFilePatterns)) {
            migratedConfig['pip-compile'].managerFilePatterns = migratedConfig['pip-compile'].managerFilePatterns.map((filePattern) => {
                const pattern = filePattern;
                if (pattern.endsWith('.in')) {
                    return pattern.replace(/\.in$/, '.txt');
                }
                if (pattern.endsWith('.in/')) {
                    return pattern.replace(/\.in\/$/, '.txt/');
                }
                return pattern.replace(/\.in\$\/$/, '.txt$/');
            });
        }
        if (is_1.default.nonEmptyArray(migratedConfig.matchManagers)) {
            if (migratedConfig.matchManagers.includes('gradle-lite')) {
                if (!migratedConfig.matchManagers.includes('gradle')) {
                    migratedConfig.matchManagers.push('gradle');
                }
                migratedConfig.matchManagers = migratedConfig.matchManagers.filter((manager) => manager !== 'gradle-lite');
            }
        }
        if (is_1.default.nonEmptyObject(migratedConfig['gradle-lite'])) {
            migratedConfig.gradle = (0, utils_1.mergeChildConfig)(migratedConfig.gradle ?? {}, migratedConfig['gradle-lite']);
        }
        delete migratedConfig['gradle-lite'];
        const isMigrated = !(0, dequal_1.dequal)(config, migratedConfig);
        if (isMigrated) {
            // recursive call in case any migrated configs need further migrating
            return {
                isMigrated,
                migratedConfig: migrateConfig(migratedConfig).migratedConfig,
            };
        }
        return { isMigrated, migratedConfig };
        /* v8 ignore next 4 -- TODO: add test */
    }
    catch (err) {
        logger_1.logger.debug({ config, err }, 'migrateConfig() error');
        throw err;
    }
}
//# sourceMappingURL=migration.js.map