/*
 * Copyright 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.ranging;

import android.content.Context;
import java.util.concurrent.Executor;

/**
 * This class provides a way to perform ranging operations such as querying the
 * device's capabilities and determining the distance and angle between the local device and a
 * remote device.
 *
 * <p>To get a {@link android.ranging.RangingManager RangingManager}, call the
 * <code>Context.getSystemService(RangingManager.class)</code>.
 *
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class RangingManager {

RangingManager() { throw new RuntimeException("Stub!"); }

/**
 * Creates a new ranging session. A ranging session enables the application
 * to perform ranging operations using available technologies such as
 * UWB (Ultra-Wideband) or WiFi RTT (Round Trip Time).
 *
 * <p>This method returns a {@link android.ranging.RangingSession RangingSession} instance, which can be
 * used to initiate, manage, and stop ranging operations. The provided
 * {@link android.ranging.RangingSession.Callback RangingSession.Callback} will be used to receive session-related
 * events, such as session start, stop, and ranging updates.
 *
 * <p>It is recommended to provide an appropriate {@link java.util.concurrent.Executor Executor} to ensure
 * that callback events are handled on a suitable thread.
 *
 * @param callback the {@link android.ranging.RangingSession.Callback RangingSession.Callback} to handle session-related events.
 *                 Must not be {@code null}.
 * @param executor the {@link java.util.concurrent.Executor Executor} on which the callback will be invoked.
 *                 Must not be {@code null}.
 * @return the {@link android.ranging.RangingSession RangingSession} instance if the session was successfully created,
 * or {@code null} if the session could not be created.
 * @throws java.lang.NullPointerException if {@code callback} or {@code executor} is null.
 * @throws java.lang.SecurityException    if the calling app does not have the necessary permissions
 *                              to create a ranging session.
 */

@android.annotation.Nullable
public android.ranging.RangingSession createRangingSession(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.ranging.RangingSession.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Registers a callback to receive ranging capabilities updates.
 *
 * @param executor The {@link java.util.concurrent.Executor Executor} on which the callback will be executed.
 *                 Must not be null.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback The {@link android.ranging.RangingManager.RangingCapabilitiesCallback RangingCapabilitiesCallback} that will handle the
 *                 capabilities updates. Must not be null.
 * @throws java.lang.NullPointerException if the {@code executor} or {@code callback} is null.
 
 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public void registerCapabilitiesCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.ranging.RangingManager.RangingCapabilitiesCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Unregisters a previously registered ranging capabilities callback.
 *
 * @param callback The {@link android.ranging.RangingManager.RangingCapabilitiesCallback RangingCapabilitiesCallback} to be unregistered.
 *                 Must not be null.
 * @throws java.lang.NullPointerException if the {@code callback} is null.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public void unregisterCapabilitiesCallback(@android.annotation.NonNull android.ranging.RangingManager.RangingCapabilitiesCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Bluetooth Channel Sounding (BT-CS) technology.
 */

public static final int BLE_CS = 1; // 0x1

/**
 * Bluetooth Low Energy (BLE) RSSI-based ranging technology.
 */

public static final int BLE_RSSI = 3; // 0x3

/**
 * Ultra-Wideband (UWB) technology.
 */

public static final int UWB = 0; // 0x0

/**
 * WiFi Round Trip Time (WiFi-RTT) technology.
 */

public static final int WIFI_NAN_RTT = 2; // 0x2
/**
 * Callback interface to receive the availabilities and capabilities of all the ranging
 * technology supported by the device.
 *
 * <p>This interface is used to asynchronously provide information about the
 * supported ranging capabilities of the device. The callback is invoked first time when
 * registered and if any capabilities are updated until it is unregistered. </p>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface RangingCapabilitiesCallback {

/**
 * Called when the ranging capabilities are available.
 *
 * @param capabilities the {@link android.ranging.RangingCapabilities RangingCapabilities} object containing
 *                     detailed information about the supported features
 *                     and limitations of the ranging technology.

 * This value cannot be {@code null}.
 */

public void onRangingCapabilities(@android.annotation.NonNull android.ranging.RangingCapabilities capabilities);
}

}

