/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.os;


/**
 * <p>
 * This class allows the caller to:
 * <ul>
 * <li>Request profiling and listen for results. Profiling types supported are: system traces,
 *     java heap dumps, heap profiles, and stack traces.</li>
 * <li>Register triggers for the system to capture profiling on the apps behalf.</li>
 * </ul>
 * </p>
 *
 * <p>
 * The {@link #requestProfiling} API can be used to begin profiling. Profiling may be ended manually
 * using the CancellationSignal provided in the request, or as a result of a timeout. The timeout
 * may be either the system default or caller defined in the parameter bundle for select types.
 * </p>
 *
 * <p>
 * The profiling results are delivered to the requesting app's data directory and a pointer to the
 * file will be received using the app provided listeners.
 * </p>
 *
 * <p>
 * Apps can provide listeners in one or both of two ways:
 * <ul>
 * <li>A request-specific listener included with the request. This will trigger only with a result
 *     from the request it was provided with.</li>
 * <li>A global listener provided by {@link #registerForAllProfilingResults}. This will be triggered
 *     for all results belonging to your app. This listener is the only way to receive results from
 *     system triggered profiling instances set up with {@link #addProfilingTriggers}.</li>
 * </ul>
 * </p>
 *
 * <p>
 * Requests are rate limited and not guaranteed to be filled. Rate limiting can be disabled for
 * local testing of {@link #requestProfiling} using the shell command
 * {@code device_config put profiling_testing rate_limiter.disabled true}
 * </p>
 *
 * <p>
 * For local testing, profiling results can be accessed more easily by enabling debug mode. This
 * will retain output files in a temporary system directory. The locations of the retained files
 * will be available in logcat. The behavior and command varies by version:
 * <ul>
 * <li>For Android versions 16 and above, debug mode will retain both unredacted (where applicable)
 * and redacted results in the temporary directory. It can be enabled with the shell command
 * {@code device_config put profiling_testing delete_temporary_results.disabled true} and disabled
 * by setting that same value back to false. Retained results are accessible on all build types.
 * </li>
 * <li>For Android version 15, debug mode will retain only the unredacted result (where applicable)
 * in the temporary directory. It can be enabled with the shell command
 * {@code device_config put profiling_testing delete_unredacted_trace.disabled true} and disabled
 * by setting that same value back to false. The retained unredacted file can only be accessed on
 * builds with root access. To access the redacted output file on an unrooted device, apps can copy
 * the file from {@code /pkg/files/profiling/file.type} to {@code /pkg/cache/file.type}.
 * </li>
 * </ul>
 * </p>
 *
 * <p>
 * In order to test profiling triggers, enable testing mode for your app with the shell command
 * {@code device_config put profiling_testing system_triggered_profiling.testing_package_name
 * com.your.app} which will:
 * <ul>
 * <li>Ensure that a background trace is running.</li>
 * <li>Allow all triggers for the provided package name to pass the system level rate limiter.
 *     This mode will continue until manually stopped with the shell command
 *     {@code device_config delete profiling_testing
 *     system_triggered_profiling.testing_package_name}.
 *     </li>
 * </ul>
 * </p>
 *
 * <p>
 * Results are redacted and contain specific information about the requesting process only.
 * </p>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class ProfilingManager {

ProfilingManager() { throw new RuntimeException("Stub!"); }

/**
 * <p>
 * Register the provided list of triggers for this process.
 * </p>
 *
 * <p>
 * Profiling triggers are system events that an app can register interest in, and then receive
 * profiling data when any of the registered triggers occur. There is no guarantee that these
 * triggers will be filled. Results, if available, will be delivered only to a global listener
 * added using {@link #registerForAllProfilingResults}.
 *</p>
 *
 * <p>
 * Only one of each trigger type can be added at a time.
 * <ul>
 * <li>If the provided list contains a trigger type that is already registered then the new one
 *     will replace the existing one.</li>
 * <li>If the provided list contains more than one trigger object for a trigger type then only
 *     one will be kept.</li>
 * </ul>
 * </p>
 *
 * <p>
 * Apps can define their own per-trigger rate limiting to help ensure they receive results
 * aligned with their needs. More details can be found at
 * {@link android.os.ProfilingTrigger.Builder#setRateLimitingPeriodHours ProfilingTrigger.Builder.setRateLimitingPeriodHours}.
 * </p>

 * @param triggers This value cannot be {@code null}.
 */

public void addProfilingTriggers(@android.annotation.NonNull java.util.List<android.os.ProfilingTrigger> triggers) { throw new RuntimeException("Stub!"); }

/** Remove all triggers for this process. */

public void clearProfilingTriggers() { throw new RuntimeException("Stub!"); }

/**
 * Register a listener to be called for all profiling results for this uid. Listeners set here
 * will be called in addition to any provided with the request.
 *
 * <p class="note"> Note: If a callback attempt fails (for example, because your app is killed
 * while a trace is in progress) re-delivery may be attempted using a listener added via this
 * method. </p>
 *
 * @param executor The executor to call back with.
 * This value cannot be {@code null}.
 * @param listener Listener to be triggered with result.

 * This value cannot be {@code null}.
 */

public void registerForAllProfilingResults(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.os.ProfilingResult> listener) { throw new RuntimeException("Stub!"); }

/**
 * Remove triggers for this process with trigger types in the provided list.
 * @param triggers This value cannot be {@code null}.

 * Value is {@link android.os.ProfilingTrigger#TRIGGER_TYPE_NONE}, {@link android.os.ProfilingTrigger#TRIGGER_TYPE_APP_FULLY_DRAWN}, or {@link android.os.ProfilingTrigger#TRIGGER_TYPE_ANR}
 */

public void removeProfilingTriggersByType(@android.annotation.NonNull int[] triggers) { throw new RuntimeException("Stub!"); }

/**
 * Request system profiling.
 *
 * <p class="note">
 *   Note: use of this API directly is not recommended for most use cases.
 *   Consider using the
 *   <a href="https://developer.android.com/reference/androidx/core/os/Profiling">higher level
 *   wrappers provided by AndroidX</a> that will construct the request correctly, supporting
 *   available options with simplified request parameters.
 * </p>
 *
 * <p>
 *   Both a listener and an executor must be set at the time of the request for the request to
 *   be considered for fulfillment. Listener/executor pairs can be set in this method, with
 *   {@link #registerForAllProfilingResults}, or both. The listener and executor must be set
 *   together, in the same call. If no listener and executor combination is set, the request
 *   will be discarded and no callback will be received.
 * </p>
 *
 * <p>
 *   Requests will be rate limited and are not guaranteed to be filled.
 * </p>
 *
 * <p>
 *   There might be a delay before profiling begins.
 *   For continuous profiling types (system tracing, stack sampling, and heap profiling),
 *   we recommend starting the collection early and stopping it with {@code cancellationSignal}
 *   immediately after the area of interest to ensure that the section you want profiled is
 *   captured.
 *   For heap dumps, we recommend testing locally to ensure that the heap dump is collected at
 *   the proper time.
 * </p>
 *
 * @param profilingType Type of profiling to collect.
 * Value is {@link android.os.ProfilingManager#PROFILING_TYPE_JAVA_HEAP_DUMP}, {@link android.os.ProfilingManager#PROFILING_TYPE_HEAP_PROFILE}, {@link android.os.ProfilingManager#PROFILING_TYPE_STACK_SAMPLING}, or {@link android.os.ProfilingManager#PROFILING_TYPE_SYSTEM_TRACE}
 * @param parameters Bundle of request related parameters. If the bundle contains any
 *                  unrecognized parameters, the request will be fail with
 *                  {@link android.os.ProfilingResult#ERROR_FAILED_INVALID_REQUEST}. If the
 *                  values for the parameters are out of supported range, the closest possible
 *                  in range value will be chosen.
 *                  Use of <a href=
 *                  "https://developer.android.com/reference/androidx/core/os/Profiling">
 *                  androidx wrappers</a> is recommended over generating this directly.
 * This value may be {@code null}.
 * @param tag Caller defined data to help identify the output.
 *                  The first 20 alphanumeric characters, plus dashes, will be lowercased
 *                  and included in the output filename.
 * This value may be {@code null}.
 * @param cancellationSignal for caller requested cancellation.
 *                  Results will be returned if available.
 *                  If this is null, the requesting app will not be able to stop the collection.
 *                  The collection will stop after timing out with either the provided
 *                  configurations or with system defaults
 * @param executor  The executor to call back with.
 *                  Will only be used for the listener provided in this method.
 *                  If this is null, and no global executor and listener combinations are
 *                  registered at the time of the request, the request will be dropped.
 * @param listener  Listener to be triggered with result. Any global listeners registered via
 *                  {@link #registerForAllProfilingResults} will also be triggered. If this is
 *                  null, and no global listener and executor combinations are registered at
 *                  the time of the request, the request will be dropped.
 */

public void requestProfiling(int profilingType, @android.annotation.Nullable android.os.Bundle parameters, @android.annotation.Nullable java.lang.String tag, @android.annotation.Nullable android.os.CancellationSignal cancellationSignal, @android.annotation.Nullable java.util.concurrent.Executor executor, @android.annotation.Nullable java.util.function.Consumer<android.os.ProfilingResult> listener) { throw new RuntimeException("Stub!"); }

/**
 * Unregister a listener that was to be called for all profiling results. If no listener is
 * provided, all listeners for this process that were not submitted with a profiling request
 * will be removed.
 *
 * @param listener Listener to unregister and no longer be triggered with the results.
 *                 Null to remove all global listeners for this uid.

 * This value may be {@code null}.
 */

public void unregisterForAllProfilingResults(@android.annotation.Nullable java.util.function.Consumer<android.os.ProfilingResult> listener) { throw new RuntimeException("Stub!"); }

/** Profiling type for {@link #requestProfiling} to request a heap profile. */

public static final int PROFILING_TYPE_HEAP_PROFILE = 2; // 0x2

/** Profiling type for {@link #requestProfiling} to request a java heap dump. */

public static final int PROFILING_TYPE_JAVA_HEAP_DUMP = 1; // 0x1

/** Profiling type for {@link #requestProfiling} to request a stack sample. */

public static final int PROFILING_TYPE_STACK_SAMPLING = 3; // 0x3

/** Profiling type for {@link #requestProfiling} to request a system trace. */

public static final int PROFILING_TYPE_SYSTEM_TRACE = 4; // 0x4
}

