/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.wifi.rtt;

import android.net.wifi.WifiSsid;

/**
 * Pre-association security negotiation (PASN) configuration.
 * <p>
 * PASN configuration in IEEE 802.11az focuses on securing the ranging process before a device
 * fully associates with a Wi-Fi network. IEEE 802.11az supports various based AKMs as in
 * {@code AKM_*} for PASN and cipher as in {@code CIPHER_*}. Password is also another input to
 * some base AKMs.
 * <p>
 * Once PASN is initiated, the AP and the client device exchange messages to authenticate each
 * other and establish security keys. This process ensures that only authorized devices can
 * participate in ranging.
 * <p>
 * After successful PASN authentication, ranging operations are performed using the established
 * secure channel. This protects the ranging measurements from eavesdropping and tampering.
 * <p>
 * The keys derived during the PASN process are used to protect the LTFs exchanged during ranging.
 * This ensures that the LTFs are encrypted and authenticated, preventing unauthorized access
 * and manipulation.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class PasnConfig implements android.os.Parcelable {

PasnConfig() { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }

/**
 * Return base AKMs (Authentication and Key Management).

 * @return Value is either <code>0</code> or a combination of {@link android.net.wifi.rtt.PasnConfig#AKM_NONE}, {@link android.net.wifi.rtt.PasnConfig#AKM_PASN}, {@link android.net.wifi.rtt.PasnConfig#AKM_SAE}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_EAP_SHA256}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_PSK_SHA256}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_EAP_SHA384}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_PSK_SHA384}, {@link android.net.wifi.rtt.PasnConfig#AKM_FILS_EAP_SHA256}, and {@link android.net.wifi.rtt.PasnConfig#AKM_FILS_EAP_SHA384}
 */

public int getBaseAkms() { throw new RuntimeException("Stub!"); }

/**
 * Return pairwise ciphers.

 * @return Value is either <code>0</code> or a combination of {@link android.net.wifi.rtt.PasnConfig#CIPHER_NONE}, {@link android.net.wifi.rtt.PasnConfig#CIPHER_CCMP_128}, {@link android.net.wifi.rtt.PasnConfig#CIPHER_CCMP_256}, {@link android.net.wifi.rtt.PasnConfig#CIPHER_GCMP_128}, and {@link android.net.wifi.rtt.PasnConfig#CIPHER_GCMP_256}
 */

public int getCiphers() { throw new RuntimeException("Stub!"); }

/**
 * Get PASN comeback cookie. See {@link android.net.wifi.rtt.PasnConfig.Builder#setPasnComebackCookie(byte[]) Builder.setPasnComebackCookie(byte[])}.
 *
 * @return This value may be {@code null}.
 */

@android.annotation.Nullable
public byte[] getPasnComebackCookie() { throw new RuntimeException("Stub!"); }

/**
 * Get password used by base AKM. If null, password is retrieved from the saved network
 * profile for the PASN authentication. See {@link #getWifiSsid()} on retrieving saved
 * network profile.
 */

@android.annotation.Nullable
public java.lang.String getPassword() { throw new RuntimeException("Stub!"); }

/**
 * Get Wifi SSID which is used to retrieve saved network profile if {@link #getPassword()}
 * is null. If Wifi SSID and password are not set and there is no saved profile corresponding to
 * the responder, unauthenticated PASN will be used if {@link android.net.wifi.rtt.RangingRequest#getSecurityMode() RangingRequest.getSecurityMode()}
 * allows. See {@code SECURITY_MODE_*} for more details.
 */

@android.annotation.Nullable
public android.net.wifi.WifiSsid getWifiSsid() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

public void writeToParcel(@android.annotation.NonNull android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

/**
 * Fast Initial Link Setup (FILS) with Extensible Authentication Protocol (EAP) and SHA-256.
 */

public static final int AKM_FILS_EAP_SHA256 = 64; // 0x40

/**
 * Fast Initial Link Setup (FILS) with Extensible Authentication Protocol (EAP) and SHA-384.
 */

public static final int AKM_FILS_EAP_SHA384 = 128; // 0x80

/**
 * Fast BSS Transition (FT) with Extensible Authentication Protocol (EAP) and SHA-256.
 */

public static final int AKM_FT_EAP_SHA256 = 4; // 0x4

/**
 * Fast BSS Transition (FT) with Extensible Authentication Protocol (EAP) and SHA-384.
 */

public static final int AKM_FT_EAP_SHA384 = 16; // 0x10

/**
 * Fast BSS Transition (FT) with Pre-Shared Key (PSK) and SHA-256.
 */

public static final int AKM_FT_PSK_SHA256 = 8; // 0x8

/**
 * Fast BSS Transition (FT) with Pre-Shared Key (PSK) and SHA-384.
 */

public static final int AKM_FT_PSK_SHA384 = 32; // 0x20

/**
 *  No authentication and key management.
 */

public static final int AKM_NONE = 0; // 0x0

/**
 * Pre-association security negotiation (PASN).
 */

public static final int AKM_PASN = 1; // 0x1

/**
 * Simultaneous authentication of equals (SAE).
 */

public static final int AKM_SAE = 2; // 0x2

/**
 * Counter Mode with Cipher Block Chaining Message Authentication Code Protocol (CCMP) with
 * 128-bit key.
 */

public static final int CIPHER_CCMP_128 = 1; // 0x1

/**
 * Counter Mode with Cipher Block Chaining Message Authentication Code Protocol (CCMP) with
 * 256-bit key.
 */

public static final int CIPHER_CCMP_256 = 2; // 0x2

/**
 * Galois/Counter Mode Protocol (GCMP) with 128-bit key.
 */

public static final int CIPHER_GCMP_128 = 4; // 0x4

/**
 * Galois/Counter Mode Protocol (GCMP) with 256-bit key.
 */

public static final int CIPHER_GCMP_256 = 8; // 0x8

/**
 * No encryption.
 */

public static final int CIPHER_NONE = 0; // 0x0

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.wifi.rtt.PasnConfig> CREATOR;
static { CREATOR = null; }
/**
 * Builder for {@link android.net.wifi.rtt.PasnConfig PasnConfig}
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

/**
 * Builder
 *
 * @param baseAkms The AKMs that PASN is configured to use. PASN will use the most secure
 *                AKM in the configuration.
 * Value is either <code>0</code> or a combination of {@link android.net.wifi.rtt.PasnConfig#AKM_NONE}, {@link android.net.wifi.rtt.PasnConfig#AKM_PASN}, {@link android.net.wifi.rtt.PasnConfig#AKM_SAE}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_EAP_SHA256}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_PSK_SHA256}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_EAP_SHA384}, {@link android.net.wifi.rtt.PasnConfig#AKM_FT_PSK_SHA384}, {@link android.net.wifi.rtt.PasnConfig#AKM_FILS_EAP_SHA256}, and {@link android.net.wifi.rtt.PasnConfig#AKM_FILS_EAP_SHA384}
 * @param ciphers  The CIPHERs that PASN is configured to use. PASN will use the most
 *                 secure CIPHER in the configuration which is applicable to the base AKM

 * Value is either <code>0</code> or a combination of {@link android.net.wifi.rtt.PasnConfig#CIPHER_NONE}, {@link android.net.wifi.rtt.PasnConfig#CIPHER_CCMP_128}, {@link android.net.wifi.rtt.PasnConfig#CIPHER_CCMP_256}, {@link android.net.wifi.rtt.PasnConfig#CIPHER_GCMP_128}, and {@link android.net.wifi.rtt.PasnConfig#CIPHER_GCMP_256}
 */

public Builder(int baseAkms, int ciphers) { throw new RuntimeException("Stub!"); }

/**
 * Builds a {@link android.net.wifi.rtt.PasnConfig PasnConfig} object.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.PasnConfig build() { throw new RuntimeException("Stub!"); }

/**
 * Set PASN comeback cookie. PASN authentication allows the station to provide comeback
 * cookie which was indicated in the {@link android.net.wifi.rtt.RangingResult RangingResult} by the AP with a deferral time.
 * <p>
 * When an AP receives a large volume of initial PASN Authentication frames, it can use
 * the comeback after field in the PASN Parameters element to indicate a deferral time
 * and optionally provide a comeback cookie which is an opaque sequence of octets. Upon
 * receiving this response, the ranging initiator (STA) must wait for the specified time
 * before retrying secure authentication, presenting the received cookie to the AP. See
 * {@link android.net.wifi.rtt.RangingResult#getPasnComebackCookie() RangingResult.getPasnComebackCookie()} and
 * {@link android.net.wifi.rtt.RangingResult#getPasnComebackAfterMillis() RangingResult.getPasnComebackAfterMillis()}.
 *
 * @param pasnComebackCookie an opaque  sequence of octets
 * This value cannot be {@code null}.
 * @return a reference to this Builder
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.PasnConfig.Builder setPasnComebackCookie(@android.annotation.NonNull byte[] pasnComebackCookie) { throw new RuntimeException("Stub!"); }

/**
 * Sets the password if needed by the base AKM of the PASN. If not set, password is
 * retrieved from the saved profile identified by the SSID. See
 * {@link #setWifiSsid(android.net.wifi.WifiSsid)}.
 *
 * Note: If password and SSID is not set, secure ranging will use unauthenticated PASN.
 *
 * @param password password string
 * This value cannot be {@code null}.
 * @return a reference to this Builder
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.PasnConfig.Builder setPassword(@android.annotation.NonNull java.lang.String password) { throw new RuntimeException("Stub!"); }

/**
 * Sets the Wi-Fi Service Set Identifier (SSID). This is used to get the saved profile to
 * retrieve password if password is not set using {@link #setPassword(java.lang.String)}.
 *
 * Note: If password and SSID is not set, secure ranging will use unauthenticated PASN.
 *
 * @param wifiSsid Wi-Fi Service Set Identifier (SSID)
 * This value cannot be {@code null}.
 * @return a reference to this Builder
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.PasnConfig.Builder setWifiSsid(@android.annotation.NonNull android.net.wifi.WifiSsid wifiSsid) { throw new RuntimeException("Stub!"); }
}

}

