/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.bluetooth;

import android.os.Build;
import java.util.List;
import java.util.UUID;

/**
 * Public API for the Bluetooth GATT Profile.
 *
 * <p>This class provides Bluetooth GATT functionality to enable communication with Bluetooth Smart
 * or Smart Ready devices.
 *
 * <p>To connect to a remote peripheral device, create a {@link android.bluetooth.BluetoothGattCallback BluetoothGattCallback} and call
 * {@link android.bluetooth.BluetoothDevice#connectGatt BluetoothDevice.connectGatt} to get a instance of this class. GATT capable devices can be
 * discovered using the Bluetooth device discovery or BLE scan process.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class BluetoothGatt implements android.bluetooth.BluetoothProfile {

BluetoothGatt() { throw new RuntimeException("Stub!"); }

/**
 * Cancels a reliable write transaction for a given device.
 *
 * <p>Calling this function will discard all queued characteristic write operations for a given
 * remote device.

 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.

 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 */

public void abortReliableWrite() { throw new RuntimeException("Stub!"); }

/**
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @deprecated Use {@link #abortReliableWrite()}
 */

@Deprecated
public void abortReliableWrite(android.bluetooth.BluetoothDevice mDevice) { throw new RuntimeException("Stub!"); }

/**
 * Initiates a reliable write transaction for a given remote device.
 *
 * <p>Once a reliable write transaction has been initiated, all calls to {@link
 * #writeCharacteristic} are sent to the remote device for verification and queued up for atomic
 * execution. The application will receive a {@link android.bluetooth.BluetoothGattCallback#onCharacteristicWrite BluetoothGattCallback.onCharacteristicWrite}
 * callback in response to every {@link #writeCharacteristic(android.bluetooth.BluetoothGattCharacteristic,byte[],int)} call and is responsible for verifying if the value has been transmitted
 * accurately.
 *
 * <p>After all characteristics have been queued up and verified, {@link #executeReliableWrite}
 * will execute all writes. If a characteristic was not written correctly, calling {@link
 * #abortReliableWrite} will cancel the current transaction without committing any values on the
 * remote device.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @return true, if the reliable write transaction has been initiated
 */

public boolean beginReliableWrite() { throw new RuntimeException("Stub!"); }

/**
 * Close this Bluetooth GATT client.
 *
 * <p>Application should call this method as early as possible after it is done with this GATT
 * client.
 
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.

 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 */

public void close() { throw new RuntimeException("Stub!"); }

/**
 * Connect back to remote device.
 *
 * <p>This method is used to re-connect to a remote device after the connection has been
 * dropped. If the device is not in range, the re-connection will be triggered once the device
 * is back in range.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @return true, if the connection attempt was initiated successfully
 */

public boolean connect() { throw new RuntimeException("Stub!"); }

/**
 * Disconnects an established connection, or cancels a connection attempt currently in progress.

 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.

 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 */

public void disconnect() { throw new RuntimeException("Stub!"); }

/**
 * Discovers services offered by a remote device as well as their characteristics and
 * descriptors.
 *
 * <p>This is an asynchronous operation. Once service discovery is completed, the {@link android.bluetooth.BluetoothGattCallback#onServicesDiscovered  } callback is triggered. If the discovery was
 * successful, the remote services can be retrieved using the {@link #getServices} function.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @return true, if the remote service discovery has been started
 */

public boolean discoverServices() { throw new RuntimeException("Stub!"); }

/**
 * Executes a reliable write transaction for a given remote device.
 *
 * <p>This function will commit all queued up characteristic write operations for a given remote
 * device.
 *
 * <p>A {@link android.bluetooth.BluetoothGattCallback#onReliableWriteCompleted BluetoothGattCallback.onReliableWriteCompleted} callback is invoked to indicate
 * whether the transaction has been executed correctly.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @return true, if the request to execute the transaction has been sent
 */

public boolean executeReliableWrite() { throw new RuntimeException("Stub!"); }

/**
 * @deprecated Not supported - please use {@link android.bluetooth.BluetoothManager#getConnectedDevices(int) BluetoothManager.getConnectedDevices(int)} with
 *     {@link android.bluetooth.BluetoothProfile#GATT BluetoothProfile.GATT} as argument
 * @throws java.lang.UnsupportedOperationException on every call
 */

@Deprecated
public java.util.List<android.bluetooth.BluetoothDevice> getConnectedDevices() { throw new RuntimeException("Stub!"); }

/**
 * @deprecated Not supported - please use {@link android.bluetooth.BluetoothManager#getConnectedDevices(int) BluetoothManager.getConnectedDevices(int)} with
 *     {@link android.bluetooth.BluetoothProfile#GATT BluetoothProfile.GATT} as argument
 * @throws java.lang.UnsupportedOperationException on every call
 */

@Deprecated
public int getConnectionState(android.bluetooth.BluetoothDevice device) { throw new RuntimeException("Stub!"); }

/**
 * Return the remote bluetooth device this GATT client targets to
 *
 * @return remote bluetooth device
 */

public android.bluetooth.BluetoothDevice getDevice() { throw new RuntimeException("Stub!"); }

/**
 * @deprecated Not supported - please use {@link android.bluetooth.BluetoothManager#getDevicesMatchingConnectionStates(int,int[])      } with {@link android.bluetooth.BluetoothProfile#GATT      } as first argument
 * @throws java.lang.UnsupportedOperationException on every call
 */

@Deprecated
public java.util.List<android.bluetooth.BluetoothDevice> getDevicesMatchingConnectionStates(int[] states) { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@link android.bluetooth.BluetoothGattService BluetoothGattService}, if the requested UUID is supported by the remote
 * device.
 *
 * <p>This function requires that service discovery has been completed for the given device.
 *
 * <p>If multiple instances of the same service (as identified by UUID) exist, the first
 * instance of the service is returned.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * @param uuid UUID of the requested service
 * @return BluetoothGattService if supported, or null if the requested service is not offered by
 *     the remote device.
 */

public android.bluetooth.BluetoothGattService getService(java.util.UUID uuid) { throw new RuntimeException("Stub!"); }

/**
 * Returns a list of GATT services offered by the remote device.
 *
 * <p>This function requires that service discovery has been completed for the given device.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * @return List of services on the remote device. Returns an empty list if service discovery has
 *     not yet been performed.
 */

public java.util.List<android.bluetooth.BluetoothGattService> getServices() { throw new RuntimeException("Stub!"); }

/**
 * Reads the requested characteristic from the associated remote device.
 *
 * <p>This is an asynchronous operation. The result of the read operation is reported by the
 * {@link android.bluetooth.BluetoothGattCallback#onCharacteristicRead(android.bluetooth.BluetoothGatt,android.bluetooth.BluetoothGattCharacteristic,byte[],int) BluetoothGattCallback.onCharacteristicRead(BluetoothGatt, BluetoothGattCharacteristic,
 * byte[], int)} callback.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param characteristic Characteristic to read from the remote device
 * @return true, if the read operation was initiated successfully
 */

public boolean readCharacteristic(android.bluetooth.BluetoothGattCharacteristic characteristic) { throw new RuntimeException("Stub!"); }

/**
 * Reads the value for a given descriptor from the associated remote device.
 *
 * <p>Once the read operation has been completed, the {@link android.bluetooth.BluetoothGattCallback#onDescriptorRead  } callback is triggered, signaling the result of the
 * operation.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param descriptor Descriptor value to read from the remote device
 * @return true, if the read operation was initiated successfully
 */

public boolean readDescriptor(android.bluetooth.BluetoothGattDescriptor descriptor) { throw new RuntimeException("Stub!"); }

/**
 * Read the current transmitter PHY and receiver PHY of the connection. The values are returned
 * in {@link android.bluetooth.BluetoothGattCallback#onPhyRead BluetoothGattCallback.onPhyRead}
 
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.

 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 */

public void readPhy() { throw new RuntimeException("Stub!"); }

/**
 * Read the RSSI for a connected remote device.
 *
 * <p>The {@link android.bluetooth.BluetoothGattCallback#onReadRemoteRssi BluetoothGattCallback.onReadRemoteRssi} callback will be invoked when the RSSI
 * value has been read.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @return true, if the RSSI value has been requested successfully
 */

public boolean readRemoteRssi() { throw new RuntimeException("Stub!"); }

/**
 * Request a connection parameter update.
 *
 * <p>This function will send a connection parameter update request to the remote device.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param connectionPriority Request a specific connection priority. Must be one of {@link android.bluetooth.BluetoothGatt#CONNECTION_PRIORITY_BALANCED      }, {@link android.bluetooth.BluetoothGatt#CONNECTION_PRIORITY_HIGH      } {@link android.bluetooth.BluetoothGatt#CONNECTION_PRIORITY_LOW_POWER      }, or {@link android.bluetooth.BluetoothGatt#CONNECTION_PRIORITY_DCK      }.
 * @throws java.lang.IllegalArgumentException If the parameters are outside of their specified range.
 */

public boolean requestConnectionPriority(int connectionPriority) { throw new RuntimeException("Stub!"); }

/**
 * Request an MTU size used for a given connection. Please note that starting from Android 14,
 * the Android Bluetooth stack requests the BLE ATT MTU to 517 bytes when the first GATT client
 * requests an MTU, and disregards all subsequent MTU requests. Check out <a
 * href="{@docRoot}about/versions/14/behavior-changes-all#mtu-set-to-517">MTU is set to 517 for
 * the first GATT client requesting an MTU</a> for more information.
 *
 * <p>When performing a write request operation (write without response), the data sent is
 * truncated to the MTU size. This function may be used to request a larger MTU size to be able
 * to send more data at once.
 *
 * <p>A {@link android.bluetooth.BluetoothGattCallback#onMtuChanged BluetoothGattCallback.onMtuChanged} callback will indicate whether this operation
 * was successful.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @return true, if the new MTU value has been requested successfully
 */

public boolean requestMtu(int mtu) { throw new RuntimeException("Stub!"); }

/**
 * Enable or disable notifications/indications for a given characteristic.
 *
 * <p>Once notifications are enabled for a characteristic, a {@link android.bluetooth.BluetoothGattCallback#onCharacteristicChanged(android.bluetooth.BluetoothGatt,android.bluetooth.BluetoothGattCharacteristic,byte[])  } callback will be triggered if the remote device indicates that the given
 * characteristic has changed.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param characteristic The characteristic for which to enable notifications
 * @param enable Set to true to enable notifications/indications
 * @return true, if the requested notification status was set successfully
 */

public boolean setCharacteristicNotification(android.bluetooth.BluetoothGattCharacteristic characteristic, boolean enable) { throw new RuntimeException("Stub!"); }

/**
 * Set the preferred connection PHY for this app. Please note that this is just a
 * recommendation, whether the PHY change will happen depends on other applications preferences,
 * local and remote controller capabilities. Controller can override these settings.
 *
 * <p>{@link android.bluetooth.BluetoothGattCallback#onPhyUpdate BluetoothGattCallback.onPhyUpdate} will be triggered as a result of this call, even
 * if no PHY change happens. It is also triggered when remote device updates the PHY.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param txPhy preferred transmitter PHY. Bitwise OR of any of {@link android.bluetooth.BluetoothDevice#PHY_LE_1M_MASK      }, {@link android.bluetooth.BluetoothDevice#PHY_LE_2M_MASK BluetoothDevice.PHY_LE_2M_MASK}, and {@link android.bluetooth.BluetoothDevice#PHY_LE_CODED_MASK      }.
 * @param rxPhy preferred receiver PHY. Bitwise OR of any of {@link android.bluetooth.BluetoothDevice#PHY_LE_1M_MASK      }, {@link android.bluetooth.BluetoothDevice#PHY_LE_2M_MASK BluetoothDevice.PHY_LE_2M_MASK}, and {@link android.bluetooth.BluetoothDevice#PHY_LE_CODED_MASK      }.
 * @param phyOptions preferred coding to use when transmitting on the LE Coded PHY. Can be one
 *     of {@link android.bluetooth.BluetoothDevice#PHY_OPTION_NO_PREFERRED BluetoothDevice.PHY_OPTION_NO_PREFERRED}, {@link android.bluetooth.BluetoothDevice#PHY_OPTION_S2 BluetoothDevice.PHY_OPTION_S2}
 *     or {@link android.bluetooth.BluetoothDevice#PHY_OPTION_S8 BluetoothDevice.PHY_OPTION_S8}
 */

public void setPreferredPhy(int txPhy, int rxPhy, int phyOptions) { throw new RuntimeException("Stub!"); }

/**
 * Writes a given characteristic and its values to the associated remote device.
 *
 * <p>Once the write operation has been completed, the {@link android.bluetooth.BluetoothGattCallback#onCharacteristicWrite  } callback is invoked, reporting the result of the
 * operation.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param characteristic Characteristic to write on the remote device
 * @return true, if the write operation was initiated successfully
 * @throws java.lang.IllegalArgumentException if characteristic or its value are null
 * @deprecated Use {@link android.bluetooth.BluetoothGatt#writeCharacteristic(android.bluetooth.BluetoothGattCharacteristic,byte[],int) BluetoothGatt.writeCharacteristic(BluetoothGattCharacteristic, byte[],
 *     int)} as this is not memory safe because it relies on a {@link android.bluetooth.BluetoothGattCharacteristic BluetoothGattCharacteristic} object whose underlying fields are subject to change outside
 *     this method.
 */

@Deprecated
public boolean writeCharacteristic(android.bluetooth.BluetoothGattCharacteristic characteristic) { throw new RuntimeException("Stub!"); }

/**
 * Writes a given characteristic and its values to the associated remote device.
 *
 * <p>Once the write operation has been completed, the {@link android.bluetooth.BluetoothGattCallback#onCharacteristicWrite  } callback is invoked, reporting the result of the
 * operation.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param characteristic Characteristic to write on the remote device
 * This value cannot be {@code null}.
 * @param value This value cannot be {@code null}.
 * @param writeType Value is {@link android.bluetooth.BluetoothGattCharacteristic#WRITE_TYPE_DEFAULT}, {@link android.bluetooth.BluetoothGattCharacteristic#WRITE_TYPE_NO_RESPONSE}, or {@link android.bluetooth.BluetoothGattCharacteristic#WRITE_TYPE_SIGNED}
 * @return whether the characteristic was successfully written to
 * Value is {@link android.bluetooth.BluetoothStatusCodes#SUCCESS}, {@link android.bluetooth.BluetoothStatusCodes#ERROR_MISSING_BLUETOOTH_CONNECT_PERMISSION}, android.bluetooth.BluetoothStatusCodes.ERROR_DEVICE_NOT_CONNECTED, {@link android.bluetooth.BluetoothStatusCodes#ERROR_PROFILE_SERVICE_NOT_BOUND}, {@link android.bluetooth.BluetoothStatusCodes#ERROR_GATT_WRITE_NOT_ALLOWED}, {@link android.bluetooth.BluetoothStatusCodes#ERROR_GATT_WRITE_REQUEST_BUSY}, or {@link android.bluetooth.BluetoothStatusCodes#ERROR_UNKNOWN}
 * @throws java.lang.IllegalArgumentException if characteristic or value are null
 */

public int writeCharacteristic(@android.annotation.NonNull android.bluetooth.BluetoothGattCharacteristic characteristic, @android.annotation.NonNull byte[] value, int writeType) { throw new RuntimeException("Stub!"); }

/**
 * Write the value of a given descriptor to the associated remote device.
 *
 * <p>A {@link android.bluetooth.BluetoothGattCallback#onDescriptorWrite BluetoothGattCallback.onDescriptorWrite} callback is triggered to report the
 * result of the write operation.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#R Build.VERSION_CODES.R} or lower, this requires the {@link android.Manifest.permission#BLUETOOTH Manifest.permission.BLUETOOTH} permission which can be gained with a simple {@code
 * <uses-permission>} manifest tag.
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param descriptor Descriptor to write to the associated remote device
 * @return true, if the write operation was initiated successfully
 * @throws java.lang.IllegalArgumentException if descriptor or its value are null
 * @deprecated Use {@link android.bluetooth.BluetoothGatt#writeDescriptor(android.bluetooth.BluetoothGattDescriptor,byte[]) BluetoothGatt.writeDescriptor(BluetoothGattDescriptor, byte[])} as
 *     this is not memory safe because it relies on a {@link android.bluetooth.BluetoothGattDescriptor BluetoothGattDescriptor} object
 *     whose underlying fields are subject to change outside this method.
 */

@Deprecated
public boolean writeDescriptor(android.bluetooth.BluetoothGattDescriptor descriptor) { throw new RuntimeException("Stub!"); }

/**
 * Write the value of a given descriptor to the associated remote device.
 *
 * <p>A {@link android.bluetooth.BluetoothGattCallback#onDescriptorWrite BluetoothGattCallback.onDescriptorWrite} callback is triggered to report the
 * result of the write operation.
 *
 * <br>
 * For apps targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES.S} or or higher, this requires the
 * {@link android.Manifest.permission#BLUETOOTH_CONNECT Manifest.permission.BLUETOOTH_CONNECT} permission which can be gained with {@link
 * android.app.Activity#requestPermissions(String[], int)}.
 * <br>
 * Requires {@link android.Manifest.permission#BLUETOOTH_CONNECT}
 * @param descriptor Descriptor to write to the associated remote device
 * This value cannot be {@code null}.
 * @param value This value cannot be {@code null}.
 * @return true, if the write operation was initiated successfully
 * Value is {@link android.bluetooth.BluetoothStatusCodes#SUCCESS}, {@link android.bluetooth.BluetoothStatusCodes#ERROR_MISSING_BLUETOOTH_CONNECT_PERMISSION}, android.bluetooth.BluetoothStatusCodes.ERROR_DEVICE_NOT_CONNECTED, {@link android.bluetooth.BluetoothStatusCodes#ERROR_PROFILE_SERVICE_NOT_BOUND}, {@link android.bluetooth.BluetoothStatusCodes#ERROR_GATT_WRITE_NOT_ALLOWED}, {@link android.bluetooth.BluetoothStatusCodes#ERROR_GATT_WRITE_REQUEST_BUSY}, or {@link android.bluetooth.BluetoothStatusCodes#ERROR_UNKNOWN}
 * @throws java.lang.IllegalArgumentException if descriptor or value are null
 */

public int writeDescriptor(@android.annotation.NonNull android.bluetooth.BluetoothGattDescriptor descriptor, @android.annotation.NonNull byte[] value) { throw new RuntimeException("Stub!"); }

/**
 * Connection parameter update - Use the connection parameters recommended by the Bluetooth SIG.
 * This is the default value if no connection parameter update is requested.
 */

public static final int CONNECTION_PRIORITY_BALANCED = 0; // 0x0

/**
 * Connection parameter update - Request the priority preferred for Digital Car Key for a lower
 * latency connection. This connection parameter will consume more power than {@link android.bluetooth.BluetoothGatt#CONNECTION_PRIORITY_BALANCED  }, so it is recommended that apps do not use this
 * unless it specifically fits their use case.
 */

public static final int CONNECTION_PRIORITY_DCK = 3; // 0x3

/**
 * Connection parameter update - Request a high priority, low latency connection. An application
 * should only request high priority connection parameters to transfer large amounts of data
 * over LE quickly. Once the transfer is complete, the application should request {@link android.bluetooth.BluetoothGatt#CONNECTION_PRIORITY_BALANCED  } connection parameters to reduce energy use.
 */

public static final int CONNECTION_PRIORITY_HIGH = 1; // 0x1

/** Connection parameter update - Request low power, reduced data rate connection parameters. */

public static final int CONNECTION_PRIORITY_LOW_POWER = 2; // 0x2

/** A remote device connection is congested. */

public static final int GATT_CONNECTION_CONGESTED = 143; // 0x8f

/**
 * GATT connection timed out, likely due to the remote device being out of range or not
 * advertising as connectable.
 */

public static final int GATT_CONNECTION_TIMEOUT = 147; // 0x93

/** A GATT operation failed, errors other than the above */

public static final int GATT_FAILURE = 257; // 0x101

/** Insufficient authentication for a given operation */

public static final int GATT_INSUFFICIENT_AUTHENTICATION = 5; // 0x5

/** Insufficient authorization for a given operation */

public static final int GATT_INSUFFICIENT_AUTHORIZATION = 8; // 0x8

/** Insufficient encryption for a given operation */

public static final int GATT_INSUFFICIENT_ENCRYPTION = 15; // 0xf

/** A write operation exceeds the maximum length of the attribute */

public static final int GATT_INVALID_ATTRIBUTE_LENGTH = 13; // 0xd

/** A read or write operation was requested with an invalid offset */

public static final int GATT_INVALID_OFFSET = 7; // 0x7

/** GATT read operation is not permitted */

public static final int GATT_READ_NOT_PERMITTED = 2; // 0x2

/** The given request is not supported */

public static final int GATT_REQUEST_NOT_SUPPORTED = 6; // 0x6

/** A GATT operation completed successfully */

public static final int GATT_SUCCESS = 0; // 0x0

/** GATT write operation is not permitted */

public static final int GATT_WRITE_NOT_PERMITTED = 3; // 0x3
}

