/*
 * Copyright (c) 2002, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.lang;

import java.util.Locale;
import java.util.Map;

/**
 * The {@code Character} class wraps a value of the primitive
 * type {@code char} in an object. An object of class
 * {@code Character} contains a single field whose type is
 * {@code char}.
 * <p>
 * In addition, this class provides several methods for determining
 * a character's category (lowercase letter, digit, etc.) and for converting
 * characters from uppercase to lowercase and vice versa.
 * <p>
 * Character information is based on the Unicode Standard
 * <p>
 * The methods and data of class {@code Character} are defined by
 * the information in the <i>UnicodeData</i> file that is part of the
 * Unicode Character Database maintained by the Unicode
 * Consortium. This file specifies various properties including name
 * and general category for every defined Unicode code point or
 * character range.
 * <p>
 * The file and its description are available from the Unicode Consortium at:
 * <ul>
 * <li><a href="http://www.unicode.org">http://www.unicode.org</a>
 * </ul>
 *
 * <h2><a id="conformance">Unicode Conformance</a></h2>
 * <p>
 * The fields and methods of class {@code Character} are defined in terms
 * of character information from the Unicode Standard, specifically the
 * <i>UnicodeData</i> file that is part of the Unicode Character Database.
 * This file specifies properties including name and category for every
 * assigned Unicode code point or character range. The file is available
 * from the Unicode Consortium at
 * <a href="http://www.unicode.org">http://www.unicode.org</a>.
 * <p>
 * Character information is based on the Unicode Standard, version 15.0.
 * <p>
 * The Java platform has supported different versions of the Unicode
 * Standard over time. Upgrades to newer versions of the Unicode Standard
 * occurred in the following Java releases, each indicating the new version:
 * <table class="striped">
 * <caption style="display:none">Shows Java releases and supported Unicode versions</caption>
 * <thead>
 * <tr><th scope="col">Java release</th>
 *     <th scope="col">Unicode version</th></tr>
 * </thead>
 * <tbody>
 * <tr><th scope="row" style="text-align:left">Java SE 20</th>
 *     <td>Unicode 15.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 19</th>
 *     <td>Unicode 14.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 15</th>
 *     <td>Unicode 13.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 13</th>
 *     <td>Unicode 12.1</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 12</th>
 *     <td>Unicode 11.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 11</th>
 *     <td>Unicode 10.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 9</th>
 *     <td>Unicode 8.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 8</th>
 *     <td>Unicode 6.2</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 7</th>
 *     <td>Unicode 6.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 5.0</th>
 *     <td>Unicode 4.0</td></tr>
 * <tr><th scope="row" style="text-align:left">Java SE 1.4</th>
 *     <td>Unicode 3.0</td></tr>
 * <tr><th scope="row" style="text-align:left">JDK 1.1</th>
 *     <td>Unicode 2.0</td></tr>
 * <tr><th scope="row" style="text-align:left">JDK 1.0.2</th>
 *     <td>Unicode 1.1.5</td></tr>
 * </tbody>
 * </table>
 * Variations from these base Unicode versions, such as recognized appendixes,
 * are documented elsewhere.
 * <h2><a id="unicode">Unicode Character Representations</a></h2>
 *
 * <p>The {@code char} data type (and therefore the value that a
 * {@code Character} object encapsulates) are based on the
 * original Unicode specification, which defined characters as
 * fixed-width 16-bit entities. The Unicode Standard has since been
 * changed to allow for characters whose representation requires more
 * than 16 bits.  The range of legal <em>code point</em>s is now
 * U+0000 to U+10FFFF, known as <em>Unicode scalar value</em>.
 * (Refer to the <a
 * href="http://www.unicode.org/reports/tr27/#notation"><i>
 * definition</i></a> of the U+<i>n</i> notation in the Unicode
 * Standard.)
 *
 * <p><a id="BMP">The set of characters from U+0000 to U+FFFF</a> is
 * sometimes referred to as the <em>Basic Multilingual Plane (BMP)</em>.
 * <a id="supplementary">Characters</a> whose code points are greater
 * than U+FFFF are called <em>supplementary character</em>s.  The Java
 * platform uses the UTF-16 representation in {@code char} arrays and
 * in the {@code String} and {@code StringBuffer} classes. In
 * this representation, supplementary characters are represented as a pair
 * of {@code char} values, the first from the <em>high-surrogates</em>
 * range, (&#92;uD800-&#92;uDBFF), the second from the
 * <em>low-surrogates</em> range (&#92;uDC00-&#92;uDFFF).
 *
 * <p>A {@code char} value, therefore, represents Basic
 * Multilingual Plane (BMP) code points, including the surrogate
 * code points, or code units of the UTF-16 encoding. An
 * {@code int} value represents all Unicode code points,
 * including supplementary code points. The lower (least significant)
 * 21 bits of {@code int} are used to represent Unicode code
 * points and the upper (most significant) 11 bits must be zero.
 * Unless otherwise specified, the behavior with respect to
 * supplementary characters and surrogate {@code char} values is
 * as follows:
 *
 * <ul>
 * <li>The methods that only accept a {@code char} value cannot support
 * supplementary characters. They treat {@code char} values from the
 * surrogate ranges as undefined characters. For example,
 * {@code Character.isLetter('\u005CuD840')} returns {@code false}, even though
 * this specific value if followed by any low-surrogate value in a string
 * would represent a letter.
 *
 * <li>The methods that accept an {@code int} value support all
 * Unicode characters, including supplementary characters. For
 * example, {@code Character.isLetter(0x2F81A)} returns
 * {@code true} because the code point value represents a letter
 * (a CJK ideograph).
 * </ul>
 *
 * <p>In the Java SE API documentation, <em>Unicode code point</em> is
 * used for character values in the range between U+0000 and U+10FFFF,
 * and <em>Unicode code unit</em> is used for 16-bit
 * {@code char} values that are code units of the <em>UTF-16</em>
 * encoding. For more information on Unicode terminology, refer to the
 * <a href="http://www.unicode.org/glossary/">Unicode Glossary</a>.
 *
 * <!-- Android-removed: paragraph on ValueBased
 * <p>This is a <a href="{@docRoot}/reference/java/lang/doc-files/ValueBased.html">value-based</a>
 * class; programmers should treat instances that are
 * {@linkplain #equals(java.lang.Object) equal} as interchangeable and should not
 * use instances for synchronization, or unpredictable behavior may
 * occur. For example, in a future release, synchronization may fail.
 * -->
 *
 * @spec https://www.unicode.org/reports/tr27 Unicode 3.1.0
 * @author  Lee Boynton
 * @author  Guy Steele
 * @author  Akira Tanaka
 * @author  Martin Buchholz
 * @author  Ulf Zibis
 * @since   1.0
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class Character implements java.lang.Comparable<java.lang.Character>, java.io.Serializable {

/**
 * Constructs a newly allocated {@code Character} object that
 * represents the specified {@code char} value.
 *
 * @param  value   the value to be represented by the
 *                  {@code Character} object.
 *
 * @deprecated
 * It is rarely appropriate to use this constructor. The static factory
 * {@link #valueOf(char)} is generally a better choice, as it is
 * likely to yield significantly better space and time performance.
 */

@Deprecated
public Character(char value) { throw new RuntimeException("Stub!"); }

/**
 * Determines the number of {@code char} values needed to
 * represent the specified character (Unicode code point). If the
 * specified character is equal to or greater than 0x10000, then
 * the method returns 2. Otherwise, the method returns 1.
 *
 * <p>This method doesn't validate the specified character to be a
 * valid Unicode code point. The caller must validate the
 * character value using {@link #isValidCodePoint(int) isValidCodePoint}
 * if necessary.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  2 if the character is a valid supplementary character; 1 otherwise.
 * @see     java.lang.Character#isSupplementaryCodePoint(int)
 * @since   1.5
 */

public static int charCount(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns the value of this {@code Character} object.
 * @return  the primitive {@code char} value represented by
 *          this object.
 */

public char charValue() { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point at the given index of the
 * {@code char} array. If the {@code char} value at
 * the given index in the {@code char} array is in the
 * high-surrogate range, the following index is less than the
 * length of the {@code char} array, and the
 * {@code char} value at the following index is in the
 * low-surrogate range, then the supplementary code point
 * corresponding to this surrogate pair is returned. Otherwise,
 * the {@code char} value at the given index is returned.
 *
 * @param a the {@code char} array
 * @param index the index to the {@code char} values (Unicode
 * code units) in the {@code char} array to be converted
 * @return the Unicode code point at the given index
 * @throws java.lang.NullPointerException if {@code a} is null.
 * @throws java.lang.IndexOutOfBoundsException if the value
 * {@code index} is negative or not less than
 * the length of the {@code char} array.
 * @since  1.5
 */

public static int codePointAt(char[] a, int index) { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point at the given index of the
 * {@code char} array, where only array elements with
 * {@code index} less than {@code limit} can be used. If
 * the {@code char} value at the given index in the
 * {@code char} array is in the high-surrogate range, the
 * following index is less than the {@code limit}, and the
 * {@code char} value at the following index is in the
 * low-surrogate range, then the supplementary code point
 * corresponding to this surrogate pair is returned. Otherwise,
 * the {@code char} value at the given index is returned.
 *
 * @param a the {@code char} array
 * @param index the index to the {@code char} values (Unicode
 * code units) in the {@code char} array to be converted
 * @param limit the index after the last array element that
 * can be used in the {@code char} array
 * @return the Unicode code point at the given index
 * @throws java.lang.NullPointerException if {@code a} is null.
 * @throws java.lang.IndexOutOfBoundsException if the {@code index}
 * argument is negative or not less than the {@code limit}
 * argument, or if the {@code limit} argument is negative or
 * greater than the length of the {@code char} array.
 * @since  1.5
 */

public static int codePointAt(char[] a, int index, int limit) { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point at the given index of the
 * {@code CharSequence}. If the {@code char} value at
 * the given index in the {@code CharSequence} is in the
 * high-surrogate range, the following index is less than the
 * length of the {@code CharSequence}, and the
 * {@code char} value at the following index is in the
 * low-surrogate range, then the supplementary code point
 * corresponding to this surrogate pair is returned. Otherwise,
 * the {@code char} value at the given index is returned.
 *
 * @param seq a sequence of {@code char} values (Unicode code
 * units)
 * @param index the index to the {@code char} values (Unicode
 * code units) in {@code seq} to be converted
 * @return the Unicode code point at the given index
 * @throws java.lang.NullPointerException if {@code seq} is null.
 * @throws java.lang.IndexOutOfBoundsException if the value
 * {@code index} is negative or not less than
 * {@link java.lang.CharSequence#length() seq.length()}.
 * @since  1.5
 */

public static int codePointAt(@androidx.annotation.RecentlyNonNull java.lang.CharSequence seq, int index) { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point preceding the given index of the
 * {@code char} array. If the {@code char} value at
 * {@code (index - 1)} in the {@code char} array is in
 * the low-surrogate range, {@code (index - 2)} is not
 * negative, and the {@code char} value at {@code (index - 2)}
 * in the {@code char} array is in the
 * high-surrogate range, then the supplementary code point
 * corresponding to this surrogate pair is returned. Otherwise,
 * the {@code char} value at {@code (index - 1)} is
 * returned.
 *
 * @param a the {@code char} array
 * @param index the index following the code point that should be returned
 * @return the Unicode code point value before the given index.
 * @throws java.lang.NullPointerException if {@code a} is null.
 * @throws java.lang.IndexOutOfBoundsException if the {@code index}
 * argument is less than 1 or greater than the length of the
 * {@code char} array
 * @since  1.5
 */

public static int codePointBefore(char[] a, int index) { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point preceding the given index of the
 * {@code char} array, where only array elements with
 * {@code index} greater than or equal to {@code start}
 * can be used. If the {@code char} value at {@code (index - 1)}
 * in the {@code char} array is in the
 * low-surrogate range, {@code (index - 2)} is not less than
 * {@code start}, and the {@code char} value at
 * {@code (index - 2)} in the {@code char} array is in
 * the high-surrogate range, then the supplementary code point
 * corresponding to this surrogate pair is returned. Otherwise,
 * the {@code char} value at {@code (index - 1)} is
 * returned.
 *
 * @param a the {@code char} array
 * @param index the index following the code point that should be returned
 * @param start the index of the first array element in the
 * {@code char} array
 * @return the Unicode code point value before the given index.
 * @throws java.lang.NullPointerException if {@code a} is null.
 * @throws java.lang.IndexOutOfBoundsException if the {@code index}
 * argument is not greater than the {@code start} argument or
 * is greater than the length of the {@code char} array, or
 * if the {@code start} argument is negative or not less than
 * the length of the {@code char} array.
 * @since  1.5
 */

public static int codePointBefore(char[] a, int index, int start) { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point preceding the given index of the
 * {@code CharSequence}. If the {@code char} value at
 * {@code (index - 1)} in the {@code CharSequence} is in
 * the low-surrogate range, {@code (index - 2)} is not
 * negative, and the {@code char} value at {@code (index - 2)}
 * in the {@code CharSequence} is in the
 * high-surrogate range, then the supplementary code point
 * corresponding to this surrogate pair is returned. Otherwise,
 * the {@code char} value at {@code (index - 1)} is
 * returned.
 *
 * @param seq the {@code CharSequence} instance
 * @param index the index following the code point that should be returned
 * @return the Unicode code point value before the given index.
 * @throws java.lang.NullPointerException if {@code seq} is null.
 * @throws java.lang.IndexOutOfBoundsException if the {@code index}
 * argument is less than 1 or greater than {@link java.lang.CharSequence#length() seq.length()}.
 * @since  1.5
 */

public static int codePointBefore(@androidx.annotation.RecentlyNonNull java.lang.CharSequence seq, int index) { throw new RuntimeException("Stub!"); }

/**
 * Returns the number of Unicode code points in a subarray of the
 * {@code char} array argument. The {@code offset}
 * argument is the index of the first {@code char} of the
 * subarray and the {@code count} argument specifies the
 * length of the subarray in {@code char}s. Unpaired
 * surrogates within the subarray count as one code point each.
 *
 * @param a the {@code char} array
 * @param offset the index of the first {@code char} in the
 * given {@code char} array
 * @param count the length of the subarray in {@code char}s
 * @return the number of Unicode code points in the specified subarray
 * @throws java.lang.NullPointerException if {@code a} is null.
 * @throws java.lang.IndexOutOfBoundsException if {@code offset} or
 * {@code count} is negative, or if {@code offset +
 * count} is larger than the length of the given array.
 * @since  1.5
 */

public static int codePointCount(char[] a, int offset, int count) { throw new RuntimeException("Stub!"); }

/**
 * Returns the number of Unicode code points in the text range of
 * the specified char sequence. The text range begins at the
 * specified {@code beginIndex} and extends to the
 * {@code char} at index {@code endIndex - 1}. Thus the
 * length (in {@code char}s) of the text range is
 * {@code endIndex-beginIndex}. Unpaired surrogates within
 * the text range count as one code point each.
 *
 * @param seq the char sequence
 * @param beginIndex the index to the first {@code char} of
 * the text range.
 * @param endIndex the index after the last {@code char} of
 * the text range.
 * @return the number of Unicode code points in the specified text
 * range
 * @throws java.lang.NullPointerException if {@code seq} is null.
 * @throws java.lang.IndexOutOfBoundsException if the
 * {@code beginIndex} is negative, or {@code endIndex}
 * is larger than the length of the given sequence, or
 * {@code beginIndex} is larger than {@code endIndex}.
 * @since  1.5
 */

public static int codePointCount(@androidx.annotation.RecentlyNonNull java.lang.CharSequence seq, int beginIndex, int endIndex) { throw new RuntimeException("Stub!"); }

/**
 * Returns the code point value of the Unicode character specified by
 * the given character name.
 * <p>
 * If a character is not assigned a name by the <i>UnicodeData</i>
 * file (part of the Unicode Character Database maintained by the Unicode
 * Consortium), its name is defined as the result of the expression:
 *
 * <blockquote>{@code
 *     Character.UnicodeBlock.of(codePoint).toString().replace('_', ' ')
 *     + " "
 *     + Integer.toHexString(codePoint).toUpperCase(Locale.ROOT);
 *
 * }</blockquote>
 * <p>
 * The {@code name} matching is case insensitive, with any leading and
 * trailing whitespace character removed.
 *
 * For the code points in the <i>UnicodeData</i> file, this method
 * recognizes the name which conforms to the name defined in the
 * "Unicode Name Property" section in the Unicode Standard. For other
 * code points, this method recognizes the name generated with
 * {@link #getName(int)} method.
 *
 * @param  name the character name
 *
 * @return the code point value of the character specified by its name.
 *
 * @throws java.lang.IllegalArgumentException if the specified {@code name}
 *         is not a valid character name.
 * @throws java.lang.NullPointerException if {@code name} is {@code null}
 *
 * @since 9
 */

public static int codePointOf(@androidx.annotation.RecentlyNonNull java.lang.String name) { throw new RuntimeException("Stub!"); }

/**
 * Compares two {@code char} values numerically.
 * The value returned is identical to what would be returned by:
 * <pre>
 *    Character.valueOf(x).compareTo(Character.valueOf(y))
 * </pre>
 *
 * @param  x the first {@code char} to compare
 * @param  y the second {@code char} to compare
 * @return the value {@code 0} if {@code x == y};
 *         a value less than {@code 0} if {@code x < y}; and
 *         a value greater than {@code 0} if {@code x > y}
 * @since 1.7
 */

public static int compare(char x, char y) { throw new RuntimeException("Stub!"); }

/**
 * Compares two {@code Character} objects numerically.
 *
 * @param   anotherCharacter   the {@code Character} to be compared.
 * @return  the value {@code 0} if the argument {@code Character}
 *          is equal to this {@code Character}; a value less than
 *          {@code 0} if this {@code Character} is numerically less
 *          than the {@code Character} argument; and a value greater than
 *          {@code 0} if this {@code Character} is numerically greater
 *          than the {@code Character} argument (unsigned comparison).
 *          Note that this is strictly a numerical comparison; it is not
 *          locale-dependent.
 * @since   1.2
 */

public int compareTo(@androidx.annotation.RecentlyNonNull java.lang.Character anotherCharacter) { throw new RuntimeException("Stub!"); }

/**
 * Returns the numeric value of the character {@code ch} in the
 * specified radix.
 * <p>
 * If the radix is not in the range {@code MIN_RADIX} &le;
 * {@code radix} &le; {@code MAX_RADIX} or if the
 * value of {@code ch} is not a valid digit in the specified
 * radix, {@code -1} is returned. A character is a valid digit
 * if at least one of the following is true:
 * <ul>
 * <li>The method {@code isDigit} is {@code true} of the character
 *     and the Unicode decimal digit value of the character (or its
 *     single-character decomposition) is less than the specified radix.
 *     In this case the decimal digit value is returned.
 * <li>The character is one of the uppercase Latin letters
 *     {@code 'A'} through {@code 'Z'} and its code is less than
 *     {@code radix + 'A' - 10}.
 *     In this case, {@code ch - 'A' + 10}
 *     is returned.
 * <li>The character is one of the lowercase Latin letters
 *     {@code 'a'} through {@code 'z'} and its code is less than
 *     {@code radix + 'a' - 10}.
 *     In this case, {@code ch - 'a' + 10}
 *     is returned.
 * <li>The character is one of the fullwidth uppercase Latin letters A
 *     ({@code '\u005CuFF21'}) through Z ({@code '\u005CuFF3A'})
 *     and its code is less than
 *     {@code radix + '\u005CuFF21' - 10}.
 *     In this case, {@code ch - '\u005CuFF21' + 10}
 *     is returned.
 * <li>The character is one of the fullwidth lowercase Latin letters a
 *     ({@code '\u005CuFF41'}) through z ({@code '\u005CuFF5A'})
 *     and its code is less than
 *     {@code radix + '\u005CuFF41' - 10}.
 *     In this case, {@code ch - '\u005CuFF41' + 10}
 *     is returned.
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #digit(int,int)} method.
 *
 * @param   ch      the character to be converted.
 * @param   radix   the radix.
 * @return  the numeric value represented by the character in the
 *          specified radix.
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Character#isDigit(char)
 */

public static int digit(char ch, int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns the numeric value of the specified character (Unicode
 * code point) in the specified radix.
 *
 * <p>If the radix is not in the range {@code MIN_RADIX} &le;
 * {@code radix} &le; {@code MAX_RADIX} or if the
 * character is not a valid digit in the specified
 * radix, {@code -1} is returned. A character is a valid digit
 * if at least one of the following is true:
 * <ul>
 * <li>The method {@link #isDigit(int) isDigit(codePoint)} is {@code true} of the character
 *     and the Unicode decimal digit value of the character (or its
 *     single-character decomposition) is less than the specified radix.
 *     In this case the decimal digit value is returned.
 * <li>The character is one of the uppercase Latin letters
 *     {@code 'A'} through {@code 'Z'} and its code is less than
 *     {@code radix + 'A' - 10}.
 *     In this case, {@code codePoint - 'A' + 10}
 *     is returned.
 * <li>The character is one of the lowercase Latin letters
 *     {@code 'a'} through {@code 'z'} and its code is less than
 *     {@code radix + 'a' - 10}.
 *     In this case, {@code codePoint - 'a' + 10}
 *     is returned.
 * <li>The character is one of the fullwidth uppercase Latin letters A
 *     ({@code '\u005CuFF21'}) through Z ({@code '\u005CuFF3A'})
 *     and its code is less than
 *     {@code radix + '\u005CuFF21' - 10}.
 *     In this case,
 *     {@code codePoint - '\u005CuFF21' + 10}
 *     is returned.
 * <li>The character is one of the fullwidth lowercase Latin letters a
 *     ({@code '\u005CuFF41'}) through z ({@code '\u005CuFF5A'})
 *     and its code is less than
 *     {@code radix + '\u005CuFF41'- 10}.
 *     In this case,
 *     {@code codePoint - '\u005CuFF41' + 10}
 *     is returned.
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be converted.
 * @param   radix   the radix.
 * @return  the numeric value represented by the character in the
 *          specified radix.
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Character#isDigit(int)
 * @since   1.5
 */

public static int digit(int codePoint, int radix) { throw new RuntimeException("Stub!"); }

/**
 * Compares this object against the specified object.
 * The result is {@code true} if and only if the argument is not
 * {@code null} and is a {@code Character} object that
 * represents the same {@code char} value as this object.
 *
 * @param   obj   the object to compare with.
 * @return  {@code true} if the objects are the same;
 *          {@code false} otherwise.
 */

public boolean equals(@androidx.annotation.RecentlyNullable java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * Determines the character representation for a specific digit in
 * the specified radix. If the value of {@code radix} is not a
 * valid radix, or the value of {@code digit} is not a valid
 * digit in the specified radix, the null character
 * ({@code '\u005Cu0000'}) is returned.
 * <p>
 * The {@code radix} argument is valid if it is greater than or
 * equal to {@code MIN_RADIX} and less than or equal to
 * {@code MAX_RADIX}. The {@code digit} argument is valid if
 * {@code 0 <= digit < radix}.
 * <p>
 * If the digit is less than 10, then
 * {@code '0' + digit} is returned. Otherwise, the value
 * {@code 'a' + digit - 10} is returned.
 *
 * @param   digit   the number to convert to a character.
 * @param   radix   the radix.
 * @return  the {@code char} representation of the specified digit
 *          in the specified radix.
 * @see     java.lang.Character#MIN_RADIX
 * @see     java.lang.Character#MAX_RADIX
 * @see     java.lang.Character#digit(char, int)
 */

public static char forDigit(int digit, int radix) { throw new RuntimeException("Stub!"); }

/**
 * Returns the Unicode directionality property for the given
 * character.  Character directionality is used to calculate the
 * visual ordering of text. The directionality value of undefined
 * {@code char} values is {@code DIRECTIONALITY_UNDEFINED}.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #getDirectionality(int)} method.
 *
 * @param  ch {@code char} for which the directionality property
 *            is requested.
 * @return the directionality property of the {@code char} value.
 *
 * @see java.lang.Character#DIRECTIONALITY_UNDEFINED
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC
 * @see java.lang.Character#DIRECTIONALITY_EUROPEAN_NUMBER
 * @see java.lang.Character#DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR
 * @see java.lang.Character#DIRECTIONALITY_ARABIC_NUMBER
 * @see java.lang.Character#DIRECTIONALITY_COMMON_NUMBER_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_NONSPACING_MARK
 * @see java.lang.Character#DIRECTIONALITY_BOUNDARY_NEUTRAL
 * @see java.lang.Character#DIRECTIONALITY_PARAGRAPH_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_SEGMENT_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_WHITESPACE
 * @see java.lang.Character#DIRECTIONALITY_OTHER_NEUTRALS
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE
 * @see java.lang.Character#DIRECTIONALITY_POP_DIRECTIONAL_FORMAT
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT_ISOLATE
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_ISOLATE
 * @see java.lang.Character#DIRECTIONALITY_FIRST_STRONG_ISOLATE
 * @see java.lang.Character#DIRECTIONALITY_POP_DIRECTIONAL_ISOLATE
 * @since 1.4
 */

public static byte getDirectionality(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Returns the Unicode directionality property for the given
 * character (Unicode code point).  Character directionality is
 * used to calculate the visual ordering of text. The
 * directionality value of undefined character is {@link
 * #DIRECTIONALITY_UNDEFINED}.
 *
 * @param   codePoint the character (Unicode code point) for which
 *          the directionality property is requested.
 * @return the directionality property of the character.
 *
 * @see java.lang.Character#DIRECTIONALITY_UNDEFINED DIRECTIONALITY_UNDEFINED
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT DIRECTIONALITY_LEFT_TO_RIGHT
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT DIRECTIONALITY_RIGHT_TO_LEFT
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC
 * @see java.lang.Character#DIRECTIONALITY_EUROPEAN_NUMBER DIRECTIONALITY_EUROPEAN_NUMBER
 * @see java.lang.Character#DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR
 * @see java.lang.Character#DIRECTIONALITY_ARABIC_NUMBER DIRECTIONALITY_ARABIC_NUMBER
 * @see java.lang.Character#DIRECTIONALITY_COMMON_NUMBER_SEPARATOR DIRECTIONALITY_COMMON_NUMBER_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_NONSPACING_MARK DIRECTIONALITY_NONSPACING_MARK
 * @see java.lang.Character#DIRECTIONALITY_BOUNDARY_NEUTRAL DIRECTIONALITY_BOUNDARY_NEUTRAL
 * @see java.lang.Character#DIRECTIONALITY_PARAGRAPH_SEPARATOR DIRECTIONALITY_PARAGRAPH_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_SEGMENT_SEPARATOR DIRECTIONALITY_SEGMENT_SEPARATOR
 * @see java.lang.Character#DIRECTIONALITY_WHITESPACE DIRECTIONALITY_WHITESPACE
 * @see java.lang.Character#DIRECTIONALITY_OTHER_NEUTRALS DIRECTIONALITY_OTHER_NEUTRALS
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE
 * @see java.lang.Character#DIRECTIONALITY_POP_DIRECTIONAL_FORMAT DIRECTIONALITY_POP_DIRECTIONAL_FORMAT
 * @see java.lang.Character#DIRECTIONALITY_LEFT_TO_RIGHT_ISOLATE DIRECTIONALITY_LEFT_TO_RIGHT_ISOLATE
 * @see java.lang.Character#DIRECTIONALITY_RIGHT_TO_LEFT_ISOLATE DIRECTIONALITY_RIGHT_TO_LEFT_ISOLATE
 * @see java.lang.Character#DIRECTIONALITY_FIRST_STRONG_ISOLATE DIRECTIONALITY_FIRST_STRONG_ISOLATE
 * @see java.lang.Character#DIRECTIONALITY_POP_DIRECTIONAL_ISOLATE DIRECTIONALITY_POP_DIRECTIONAL_ISOLATE
 * @since    1.5
 */

public static byte getDirectionality(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of the specified character
 * {@code codePoint}, or null if the code point is
 * {@link #UNASSIGNED unassigned}.
 * <p>
 * If the specified character is not assigned a name by
 * the <i>UnicodeData</i> file (part of the Unicode Character
 * Database maintained by the Unicode Consortium), the returned
 * name is the same as the result of the expression:
 *
 * <blockquote>{@code
 *     Character.UnicodeBlock.of(codePoint).toString().replace('_', ' ')
 *     + " "
 *     + Integer.toHexString(codePoint).toUpperCase(Locale.ROOT);
 *
 * }</blockquote>
 *
 * For the {@code codePoint}s in the <i>UnicodeData</i> file, the name
 * returned by this method follows the naming scheme in the
 * "Unicode Name Property" section of the Unicode Standard. For other
 * code points, such as Hangul/Ideographs, The name generation rule above
 * differs from the one defined in the Unicode Standard.
 *
 * @param  codePoint the character (Unicode code point)
 *
 * @return the name of the specified character, or null if
 *         the code point is unassigned.
 *
 * @throws java.lang.IllegalArgumentException if the specified
 *            {@code codePoint} is not a valid Unicode
 *            code point.
 *
 * @since 1.7
 */

@androidx.annotation.RecentlyNullable
public static java.lang.String getName(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@code int} value that the specified Unicode
 * character represents. For example, the character
 * {@code '\u005Cu216C'} (the roman numeral fifty) will return
 * an int with a value of 50.
 * <p>
 * The letters A-Z in their uppercase ({@code '\u005Cu0041'} through
 * {@code '\u005Cu005A'}), lowercase
 * ({@code '\u005Cu0061'} through {@code '\u005Cu007A'}), and
 * full width variant ({@code '\u005CuFF21'} through
 * {@code '\u005CuFF3A'} and {@code '\u005CuFF41'} through
 * {@code '\u005CuFF5A'}) forms have numeric values from 10
 * through 35. This is independent of the Unicode specification,
 * which does not assign numeric values to these {@code char}
 * values.
 * <p>
 * If the character does not have a numeric value, then -1 is returned.
 * If the character has a numeric value that cannot be represented as a
 * nonnegative integer (for example, a fractional value), then -2
 * is returned.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #getNumericValue(int)} method.
 *
 * @param   ch      the character to be converted.
 * @return  the numeric value of the character, as a nonnegative {@code int}
 *          value; -2 if the character has a numeric value but the value
 *          can not be represented as a nonnegative {@code int} value;
 *          -1 if the character has no numeric value.
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Character#isDigit(char)
 * @since   1.1
 */

public static int getNumericValue(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@code int} value that the specified
 * character (Unicode code point) represents. For example, the character
 * {@code '\u005Cu216C'} (the Roman numeral fifty) will return
 * an {@code int} with a value of 50.
 * <p>
 * The letters A-Z in their uppercase ({@code '\u005Cu0041'} through
 * {@code '\u005Cu005A'}), lowercase
 * ({@code '\u005Cu0061'} through {@code '\u005Cu007A'}), and
 * full width variant ({@code '\u005CuFF21'} through
 * {@code '\u005CuFF3A'} and {@code '\u005CuFF41'} through
 * {@code '\u005CuFF5A'}) forms have numeric values from 10
 * through 35. This is independent of the Unicode specification,
 * which does not assign numeric values to these {@code char}
 * values.
 * <p>
 * If the character does not have a numeric value, then -1 is returned.
 * If the character has a numeric value that cannot be represented as a
 * nonnegative integer (for example, a fractional value), then -2
 * is returned.
 *
 * @param   codePoint the character (Unicode code point) to be converted.
 * @return  the numeric value of the character, as a nonnegative {@code int}
 *          value; -2 if the character has a numeric value but the value
 *          can not be represented as a nonnegative {@code int} value;
 *          -1 if the character has no numeric value.
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Character#isDigit(int)
 * @since   1.5
 */

public static int getNumericValue(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns a value indicating a character's general category.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #getType(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return  a value of type {@code int} representing the
 *          character's general category.
 * @see     java.lang.Character#COMBINING_SPACING_MARK
 * @see     java.lang.Character#CONNECTOR_PUNCTUATION
 * @see     java.lang.Character#CONTROL
 * @see     java.lang.Character#CURRENCY_SYMBOL
 * @see     java.lang.Character#DASH_PUNCTUATION
 * @see     java.lang.Character#DECIMAL_DIGIT_NUMBER
 * @see     java.lang.Character#ENCLOSING_MARK
 * @see     java.lang.Character#END_PUNCTUATION
 * @see     java.lang.Character#FINAL_QUOTE_PUNCTUATION
 * @see     java.lang.Character#FORMAT
 * @see     java.lang.Character#INITIAL_QUOTE_PUNCTUATION
 * @see     java.lang.Character#LETTER_NUMBER
 * @see     java.lang.Character#LINE_SEPARATOR
 * @see     java.lang.Character#LOWERCASE_LETTER
 * @see     java.lang.Character#MATH_SYMBOL
 * @see     java.lang.Character#MODIFIER_LETTER
 * @see     java.lang.Character#MODIFIER_SYMBOL
 * @see     java.lang.Character#NON_SPACING_MARK
 * @see     java.lang.Character#OTHER_LETTER
 * @see     java.lang.Character#OTHER_NUMBER
 * @see     java.lang.Character#OTHER_PUNCTUATION
 * @see     java.lang.Character#OTHER_SYMBOL
 * @see     java.lang.Character#PARAGRAPH_SEPARATOR
 * @see     java.lang.Character#PRIVATE_USE
 * @see     java.lang.Character#SPACE_SEPARATOR
 * @see     java.lang.Character#START_PUNCTUATION
 * @see     java.lang.Character#SURROGATE
 * @see     java.lang.Character#TITLECASE_LETTER
 * @see     java.lang.Character#UNASSIGNED
 * @see     java.lang.Character#UPPERCASE_LETTER
 * @since   1.1
 */

public static int getType(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Returns a value indicating a character's general category.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  a value of type {@code int} representing the
 *          character's general category.
 * @see     java.lang.Character#COMBINING_SPACING_MARK COMBINING_SPACING_MARK
 * @see     java.lang.Character#CONNECTOR_PUNCTUATION CONNECTOR_PUNCTUATION
 * @see     java.lang.Character#CONTROL CONTROL
 * @see     java.lang.Character#CURRENCY_SYMBOL CURRENCY_SYMBOL
 * @see     java.lang.Character#DASH_PUNCTUATION DASH_PUNCTUATION
 * @see     java.lang.Character#DECIMAL_DIGIT_NUMBER DECIMAL_DIGIT_NUMBER
 * @see     java.lang.Character#ENCLOSING_MARK ENCLOSING_MARK
 * @see     java.lang.Character#END_PUNCTUATION END_PUNCTUATION
 * @see     java.lang.Character#FINAL_QUOTE_PUNCTUATION FINAL_QUOTE_PUNCTUATION
 * @see     java.lang.Character#FORMAT FORMAT
 * @see     java.lang.Character#INITIAL_QUOTE_PUNCTUATION INITIAL_QUOTE_PUNCTUATION
 * @see     java.lang.Character#LETTER_NUMBER LETTER_NUMBER
 * @see     java.lang.Character#LINE_SEPARATOR LINE_SEPARATOR
 * @see     java.lang.Character#LOWERCASE_LETTER LOWERCASE_LETTER
 * @see     java.lang.Character#MATH_SYMBOL MATH_SYMBOL
 * @see     java.lang.Character#MODIFIER_LETTER MODIFIER_LETTER
 * @see     java.lang.Character#MODIFIER_SYMBOL MODIFIER_SYMBOL
 * @see     java.lang.Character#NON_SPACING_MARK NON_SPACING_MARK
 * @see     java.lang.Character#OTHER_LETTER OTHER_LETTER
 * @see     java.lang.Character#OTHER_NUMBER OTHER_NUMBER
 * @see     java.lang.Character#OTHER_PUNCTUATION OTHER_PUNCTUATION
 * @see     java.lang.Character#OTHER_SYMBOL OTHER_SYMBOL
 * @see     java.lang.Character#PARAGRAPH_SEPARATOR PARAGRAPH_SEPARATOR
 * @see     java.lang.Character#PRIVATE_USE PRIVATE_USE
 * @see     java.lang.Character#SPACE_SEPARATOR SPACE_SEPARATOR
 * @see     java.lang.Character#START_PUNCTUATION START_PUNCTUATION
 * @see     java.lang.Character#SURROGATE SURROGATE
 * @see     java.lang.Character#TITLECASE_LETTER TITLECASE_LETTER
 * @see     java.lang.Character#UNASSIGNED UNASSIGNED
 * @see     java.lang.Character#UPPERCASE_LETTER UPPERCASE_LETTER
 * @since   1.5
 */

public static int getType(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns a hash code for this {@code Character}; equal to the result
 * of invoking {@code charValue()}.
 *
 * @return a hash code value for this {@code Character}
 */

public int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Returns a hash code for a {@code char} value; compatible with
 * {@code Character.hashCode()}.
 *
 * @since 1.8
 *
 * @param value The {@code char} for which to return a hash code.
 * @return a hash code value for a {@code char} value.
 */

public static int hashCode(char value) { throw new RuntimeException("Stub!"); }

/**
 * Returns the leading surrogate (a
 * <a href="http://www.unicode.org/glossary/#high_surrogate_code_unit">
 * high surrogate code unit</a>) of the
 * <a href="http://www.unicode.org/glossary/#surrogate_pair">
 * surrogate pair</a>
 * representing the specified supplementary character (Unicode
 * code point) in the UTF-16 encoding.  If the specified character
 * is not a
 * <a href="Character.html#supplementary">supplementary character</a>,
 * an unspecified {@code char} is returned.
 *
 * <p>If
 * {@link #isSupplementaryCodePoint isSupplementaryCodePoint(x)}
 * is {@code true}, then
 * {@link #isHighSurrogate isHighSurrogate}{@code (highSurrogate(x))} and
 * {@link #toCodePoint toCodePoint}{@code (highSurrogate(x), }{@link #lowSurrogate lowSurrogate}{@code (x)) == x}
 * are also always {@code true}.
 *
 * @param   codePoint a supplementary character (Unicode code point)
 * @return  the leading surrogate code unit used to represent the
 *          character in the UTF-16 encoding
 * @since   1.7
 */

public static char highSurrogate(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is alphabetic.
 * <p>
 * A character is considered to be alphabetic if its general category type,
 * provided by {@link java.lang.Character#getType(int) getType(codePoint)}, is any of
 * the following:
 * <ul>
 * <li> {@code UPPERCASE_LETTER}
 * <li> {@code LOWERCASE_LETTER}
 * <li> {@code TITLECASE_LETTER}
 * <li> {@code MODIFIER_LETTER}
 * <li> {@code OTHER_LETTER}
 * <li> {@code LETTER_NUMBER}
 * </ul>
 * or it has contributory property Other_Alphabetic as defined by the
 * Unicode Standard.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a Unicode alphabet
 *          character, {@code false} otherwise.
 * @since   1.7
 */

public static boolean isAlphabetic(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines whether the specified character (Unicode code point)
 * is in the <a href="#BMP">Basic Multilingual Plane (BMP)</a>.
 * Such code points can be represented using a single {@code char}.
 *
 * @param  codePoint the character (Unicode code point) to be to
 * @return {@code true} if the specified code point is between
 *         {@link #MIN_VALUE} and {@link #MAX_VALUE} inclusive;
 *         {@code false} otherwise.
 * @since  1.7
 */

public static boolean isBmpCodePoint(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if a character is defined in Unicode.
 * <p>
 * A character is defined if at least one of the following is true:
 * <ul>
 * <li>It has an entry in the UnicodeData file.
 * <li>It has a value in a range defined by the UnicodeData file.
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isDefined(int)} method.
 *
 * @param   ch   the character to be tested
 * @return  {@code true} if the character has a defined meaning
 *          in Unicode; {@code false} otherwise.
 * @see     java.lang.Character#isDigit(char)
 * @see     java.lang.Character#isLetter(char)
 * @see     java.lang.Character#isLetterOrDigit(char)
 * @see     java.lang.Character#isLowerCase(char)
 * @see     java.lang.Character#isTitleCase(char)
 * @see     java.lang.Character#isUpperCase(char)
 * @since   1.0.2
 */

public static boolean isDefined(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if a character (Unicode code point) is defined in Unicode.
 * <p>
 * A character is defined if at least one of the following is true:
 * <ul>
 * <li>It has an entry in the UnicodeData file.
 * <li>It has a value in a range defined by the UnicodeData file.
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character has a defined meaning
 *          in Unicode; {@code false} otherwise.
 * @see     java.lang.Character#isDigit(int)
 * @see     java.lang.Character#isLetter(int)
 * @see     java.lang.Character#isLetterOrDigit(int)
 * @see     java.lang.Character#isLowerCase(int)
 * @see     java.lang.Character#isTitleCase(int)
 * @see     java.lang.Character#isUpperCase(int)
 * @since   1.5
 */

public static boolean isDefined(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is a digit.
 * <p>
 * A character is a digit if its general category type, provided
 * by {@code Character.getType(ch)}, is
 * {@code DECIMAL_DIGIT_NUMBER}.
 * <p>
 * Some Unicode character ranges that contain digits:
 * <ul>
 * <li>{@code '\u005Cu0030'} through {@code '\u005Cu0039'},
 *     ISO-LATIN-1 digits ({@code '0'} through {@code '9'})
 * <li>{@code '\u005Cu0660'} through {@code '\u005Cu0669'},
 *     Arabic-Indic digits
 * <li>{@code '\u005Cu06F0'} through {@code '\u005Cu06F9'},
 *     Extended Arabic-Indic digits
 * <li>{@code '\u005Cu0966'} through {@code '\u005Cu096F'},
 *     Devanagari digits
 * <li>{@code '\u005CuFF10'} through {@code '\u005CuFF19'},
 *     Fullwidth digits
 * </ul>
 *
 * Many other character ranges contain digits as well.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isDigit(int)} method.
 *
 * @param   ch   the character to be tested.
 * @return  {@code true} if the character is a digit;
 *          {@code false} otherwise.
 * @see     java.lang.Character#digit(char, int)
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Character#getType(char)
 */

public static boolean isDigit(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a digit.
 * <p>
 * A character is a digit if its general category type, provided
 * by {@link java.lang.Character#getType(int) getType(codePoint)}, is
 * {@code DECIMAL_DIGIT_NUMBER}.
 * <p>
 * Some Unicode character ranges that contain digits:
 * <ul>
 * <li>{@code '\u005Cu0030'} through {@code '\u005Cu0039'},
 *     ISO-LATIN-1 digits ({@code '0'} through {@code '9'})
 * <li>{@code '\u005Cu0660'} through {@code '\u005Cu0669'},
 *     Arabic-Indic digits
 * <li>{@code '\u005Cu06F0'} through {@code '\u005Cu06F9'},
 *     Extended Arabic-Indic digits
 * <li>{@code '\u005Cu0966'} through {@code '\u005Cu096F'},
 *     Devanagari digits
 * <li>{@code '\u005CuFF10'} through {@code '\u005CuFF19'},
 *     Fullwidth digits
 * </ul>
 *
 * Many other character ranges contain digits as well.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a digit;
 *          {@code false} otherwise.
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Character#getType(int)
 * @since   1.5
 */

public static boolean isDigit(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is an Emoji.
 * <p>
 * A character is considered to be an Emoji if and only if it has the {@code Emoji}
 * property, defined in
 * <a href="https://unicode.org/reports/tr51/#Emoji_Properties_and_Data_Files">
 * Unicode Emoji (Technical Standard #51)</a>.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an Emoji;
 *          {@code false} otherwise.
 * @since   21
 */

public static boolean isEmoji(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is an
 * Emoji Component.
 * <p>
 * A character is considered to be an Emoji Component if and only if it has
 * the {@code Emoji_Component} property, defined in
 * <a href="https://unicode.org/reports/tr51/#Emoji_Properties_and_Data_Files">
 * Unicode Emoji (Technical Standard #51)</a>.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an Emoji Component;
 *          {@code false} otherwise.
 * @since   21
 */

public static boolean isEmojiComponent(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is an
 * Emoji Modifier.
 * <p>
 * A character is considered to be an Emoji Modifier if and only if it has
 * the {@code Emoji_Modifier} property, defined in
 * <a href="https://unicode.org/reports/tr51/#Emoji_Properties_and_Data_Files">
 * Unicode Emoji (Technical Standard #51)</a>.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an Emoji Modifier;
 *          {@code false} otherwise.
 * @since   21
 */

public static boolean isEmojiModifier(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is an
 * Emoji Modifier Base.
 * <p>
 * A character is considered to be an Emoji Modifier Base if and only if it has
 * the {@code Emoji_Modifier_Base} property, defined in
 * <a href="https://unicode.org/reports/tr51/#Emoji_Properties_and_Data_Files">
 * Unicode Emoji (Technical Standard #51)</a>.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an Emoji Modifier Base;
 *          {@code false} otherwise.
 * @since   21
 */

public static boolean isEmojiModifierBase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) has the
 * Emoji Presentation property by default.
 * <p>
 * A character is considered to have the Emoji Presentation property if and
 * only if it has the {@code Emoji_Presentation} property, defined in
 * <a href="https://unicode.org/reports/tr51/#Emoji_Properties_and_Data_Files">
 * Unicode Emoji (Technical Standard #51)</a>.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character has the Emoji Presentation
 *          property; {@code false} otherwise.
 * @since   21
 */

public static boolean isEmojiPresentation(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is
 * an Extended Pictographic.
 * <p>
 * A character is considered to be an Extended Pictographic if and only if it has
 * the {@code Extended_Pictographic} property, defined in
 * <a href="https://unicode.org/reports/tr51/#Emoji_Properties_and_Data_Files">
 * Unicode Emoji (Technical Standard #51)</a>.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an Extended Pictographic;
 *          {@code false} otherwise.
 * @since   21
 */

public static boolean isExtendedPictographic(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the given {@code char} value is a
 * <a href="http://www.unicode.org/glossary/#high_surrogate_code_unit">
 * Unicode high-surrogate code unit</a>
 * (also known as <i>leading-surrogate code unit</i>).
 *
 * <p>Such values do not represent characters by themselves,
 * but are used in the representation of
 * <a href="#supplementary">supplementary characters</a>
 * in the UTF-16 encoding.
 *
 * @param  ch the {@code char} value to be tested.
 * @return {@code true} if the {@code char} value is between
 *         {@link #MIN_HIGH_SURROGATE} and
 *         {@link #MAX_HIGH_SURROGATE} inclusive;
 *         {@code false} otherwise.
 * @see    java.lang.Character#isLowSurrogate(char)
 * @see    java.lang.Character.UnicodeBlock#of(int)
 * @since  1.5
 */

public static boolean isHighSurrogate(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is an ISO control
 * character.  A character is considered to be an ISO control
 * character if its code is in the range {@code '\u005Cu0000'}
 * through {@code '\u005Cu001F'} or in the range
 * {@code '\u005Cu007F'} through {@code '\u005Cu009F'}.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isISOControl(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return  {@code true} if the character is an ISO control character;
 *          {@code false} otherwise.
 *
 * @see     java.lang.Character#isSpaceChar(char)
 * @see     java.lang.Character#isWhitespace(char)
 * @since   1.1
 */

public static boolean isISOControl(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the referenced character (Unicode code point) is an ISO control
 * character.  A character is considered to be an ISO control
 * character if its code is in the range {@code '\u005Cu0000'}
 * through {@code '\u005Cu001F'} or in the range
 * {@code '\u005Cu007F'} through {@code '\u005Cu009F'}.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an ISO control character;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isSpaceChar(int)
 * @see     java.lang.Character#isWhitespace(int)
 * @since   1.5
 */

public static boolean isISOControl(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character should be regarded as
 * an ignorable character in a Java identifier or a Unicode identifier.
 * <p>
 * The following Unicode characters are ignorable in a Java identifier
 * or a Unicode identifier:
 * <ul>
 * <li>ISO control characters that are not whitespace
 * <ul>
 * <li>{@code '\u005Cu0000'} through {@code '\u005Cu0008'}
 * <li>{@code '\u005Cu000E'} through {@code '\u005Cu001B'}
 * <li>{@code '\u005Cu007F'} through {@code '\u005Cu009F'}
 * </ul>
 *
 * <li>all characters that have the {@code FORMAT} general
 * category value
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isIdentifierIgnorable(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return  {@code true} if the character is an ignorable control
 *          character that may be part of a Java or Unicode identifier;
 *           {@code false} otherwise.
 * @see     java.lang.Character#isJavaIdentifierPart(char)
 * @see     java.lang.Character#isUnicodeIdentifierPart(char)
 * @since   1.1
 */

public static boolean isIdentifierIgnorable(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) should be regarded as
 * an ignorable character in a Java identifier or a Unicode identifier.
 * <p>
 * The following Unicode characters are ignorable in a Java identifier
 * or a Unicode identifier:
 * <ul>
 * <li>ISO control characters that are not whitespace
 * <ul>
 * <li>{@code '\u005Cu0000'} through {@code '\u005Cu0008'}
 * <li>{@code '\u005Cu000E'} through {@code '\u005Cu001B'}
 * <li>{@code '\u005Cu007F'} through {@code '\u005Cu009F'}
 * </ul>
 *
 * <li>all characters that have the {@code FORMAT} general
 * category value
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is an ignorable control
 *          character that may be part of a Java or Unicode identifier;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isJavaIdentifierPart(int)
 * @see     java.lang.Character#isUnicodeIdentifierPart(int)
 * @since   1.5
 */

public static boolean isIdentifierIgnorable(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a CJKV
 * (Chinese, Japanese, Korean and Vietnamese) ideograph, as defined by
 * the Unicode Standard.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a Unicode ideograph
 *          character, {@code false} otherwise.
 * @since   1.7
 */

public static boolean isIdeographic(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character may be part of a Java
 * identifier as other than the first character.
 * <p>
 * A character may be part of a Java identifier if any of the following
 * conditions are true:
 * <ul>
 * <li>  it is a letter
 * <li>  it is a currency symbol (such as {@code '$'})
 * <li>  it is a connecting punctuation character (such as {@code '_'})
 * <li>  it is a digit
 * <li>  it is a numeric letter (such as a Roman numeral character)
 * <li>  it is a combining mark
 * <li>  it is a non-spacing mark
 * <li> {@code isIdentifierIgnorable} returns
 * {@code true} for the character
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isJavaIdentifierPart(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return {@code true} if the character may be part of a
 *          Java identifier; {@code false} otherwise.
 * @see     java.lang.Character#isIdentifierIgnorable(char)
 * @see     java.lang.Character#isJavaIdentifierStart(char)
 * @see     java.lang.Character#isLetterOrDigit(char)
 * @see     java.lang.Character#isUnicodeIdentifierPart(char)
 * @since   1.1
 */

public static boolean isJavaIdentifierPart(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the character (Unicode code point) may be part of a Java
 * identifier as other than the first character.
 * <p>
 * A character may be part of a Java identifier if any of the following
 * conditions are true:
 * <ul>
 * <li>  it is a letter
 * <li>  it is a currency symbol (such as {@code '$'})
 * <li>  it is a connecting punctuation character (such as {@code '_'})
 * <li>  it is a digit
 * <li>  it is a numeric letter (such as a Roman numeral character)
 * <li>  it is a combining mark
 * <li>  it is a non-spacing mark
 * <li> {@link #isIdentifierIgnorable(int)
 * isIdentifierIgnorable(codePoint)} returns {@code true} for
 * the code point
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return {@code true} if the character may be part of a
 *          Java identifier; {@code false} otherwise.
 * @see     java.lang.Character#isIdentifierIgnorable(int)
 * @see     java.lang.Character#isJavaIdentifierStart(int)
 * @see     java.lang.Character#isLetterOrDigit(int)
 * @see     java.lang.Character#isUnicodeIdentifierPart(int)
 * @since   1.5
 */

public static boolean isJavaIdentifierPart(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is
 * permissible as the first character in a Java identifier.
 * <p>
 * A character may start a Java identifier if and only if
 * one of the following conditions is true:
 * <ul>
 * <li> {@link #isLetter(char) isLetter(ch)} returns {@code true}
 * <li> {@link #getType(char) getType(ch)} returns {@code LETTER_NUMBER}
 * <li> {@code ch} is a currency symbol (such as {@code '$'})
 * <li> {@code ch} is a connecting punctuation character (such as {@code '_'}).
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isJavaIdentifierStart(int)} method.
 *
 * @param   ch the character to be tested.
 * @return  {@code true} if the character may start a Java identifier;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isJavaIdentifierPart(char)
 * @see     java.lang.Character#isLetter(char)
 * @see     java.lang.Character#isUnicodeIdentifierStart(char)
 * @since   1.1
 */

public static boolean isJavaIdentifierStart(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the character (Unicode code point) is
 * permissible as the first character in a Java identifier.
 * <p>
 * A character may start a Java identifier if and only if
 * one of the following conditions is true:
 * <ul>
 * <li> {@link #isLetter(int) isLetter(codePoint)}
 *      returns {@code true}
 * <li> {@link #getType(int) getType(codePoint)}
 *      returns {@code LETTER_NUMBER}
 * <li> the referenced character is a currency symbol (such as {@code '$'})
 * <li> the referenced character is a connecting punctuation character
 *      (such as {@code '_'}).
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character may start a Java identifier;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isJavaIdentifierPart(int)
 * @see     java.lang.Character#isLetter(int)
 * @see     java.lang.Character#isUnicodeIdentifierStart(int)
 * @since   1.5
 */

public static boolean isJavaIdentifierStart(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is permissible as the first
 * character in a Java identifier.
 * <p>
 * A character may start a Java identifier if and only if
 * one of the following conditions is true:
 * <ul>
 * <li> {@link #isLetter(char) isLetter(ch)} returns {@code true}
 * <li> {@link #getType(char) getType(ch)} returns {@code LETTER_NUMBER}
 * <li> {@code ch} is a currency symbol (such as {@code '$'})
 * <li> {@code ch} is a connecting punctuation character (such as {@code '_'}).
 * </ul>
 *
 * @param   ch the character to be tested.
 * @return  {@code true} if the character may start a Java
 *          identifier; {@code false} otherwise.
 * @see     java.lang.Character#isJavaLetterOrDigit(char)
 * @see     java.lang.Character#isJavaIdentifierStart(char)
 * @see     java.lang.Character#isJavaIdentifierPart(char)
 * @see     java.lang.Character#isLetter(char)
 * @see     java.lang.Character#isLetterOrDigit(char)
 * @see     java.lang.Character#isUnicodeIdentifierStart(char)
 * @since   1.0.2
 * @deprecated Replaced by isJavaIdentifierStart(char).
 */

@Deprecated
public static boolean isJavaLetter(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character may be part of a Java
 * identifier as other than the first character.
 * <p>
 * A character may be part of a Java identifier if and only if one
 * of the following conditions is true:
 * <ul>
 * <li>  it is a letter
 * <li>  it is a currency symbol (such as {@code '$'})
 * <li>  it is a connecting punctuation character (such as {@code '_'})
 * <li>  it is a digit
 * <li>  it is a numeric letter (such as a Roman numeral character)
 * <li>  it is a combining mark
 * <li>  it is a non-spacing mark
 * <li> {@code isIdentifierIgnorable} returns
 * {@code true} for the character.
 * </ul>
 *
 * @param   ch the character to be tested.
 * @return  {@code true} if the character may be part of a
 *          Java identifier; {@code false} otherwise.
 * @see     java.lang.Character#isJavaLetter(char)
 * @see     java.lang.Character#isJavaIdentifierStart(char)
 * @see     java.lang.Character#isJavaIdentifierPart(char)
 * @see     java.lang.Character#isLetter(char)
 * @see     java.lang.Character#isLetterOrDigit(char)
 * @see     java.lang.Character#isUnicodeIdentifierPart(char)
 * @see     java.lang.Character#isIdentifierIgnorable(char)
 * @since   1.0.2
 * @deprecated Replaced by isJavaIdentifierPart(char).
 */

@Deprecated
public static boolean isJavaLetterOrDigit(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is a letter.
 * <p>
 * A character is considered to be a letter if its general
 * category type, provided by {@code Character.getType(ch)},
 * is any of the following:
 * <ul>
 * <li> {@code UPPERCASE_LETTER}
 * <li> {@code LOWERCASE_LETTER}
 * <li> {@code TITLECASE_LETTER}
 * <li> {@code MODIFIER_LETTER}
 * <li> {@code OTHER_LETTER}
 * </ul>
 *
 * Not all letters have case. Many characters are
 * letters but are neither uppercase nor lowercase nor titlecase.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isLetter(int)} method.
 *
 * @param   ch   the character to be tested.
 * @return  {@code true} if the character is a letter;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isDigit(char)
 * @see     java.lang.Character#isJavaIdentifierStart(char)
 * @see     java.lang.Character#isJavaLetter(char)
 * @see     java.lang.Character#isJavaLetterOrDigit(char)
 * @see     java.lang.Character#isLetterOrDigit(char)
 * @see     java.lang.Character#isLowerCase(char)
 * @see     java.lang.Character#isTitleCase(char)
 * @see     java.lang.Character#isUnicodeIdentifierStart(char)
 * @see     java.lang.Character#isUpperCase(char)
 */

public static boolean isLetter(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a letter.
 * <p>
 * A character is considered to be a letter if its general
 * category type, provided by {@link java.lang.Character#getType(int) getType(codePoint)},
 * is any of the following:
 * <ul>
 * <li> {@code UPPERCASE_LETTER}
 * <li> {@code LOWERCASE_LETTER}
 * <li> {@code TITLECASE_LETTER}
 * <li> {@code MODIFIER_LETTER}
 * <li> {@code OTHER_LETTER}
 * </ul>
 *
 * Not all letters have case. Many characters are
 * letters but are neither uppercase nor lowercase nor titlecase.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a letter;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isDigit(int)
 * @see     java.lang.Character#isJavaIdentifierStart(int)
 * @see     java.lang.Character#isLetterOrDigit(int)
 * @see     java.lang.Character#isLowerCase(int)
 * @see     java.lang.Character#isTitleCase(int)
 * @see     java.lang.Character#isUnicodeIdentifierStart(int)
 * @see     java.lang.Character#isUpperCase(int)
 * @since   1.5
 */

public static boolean isLetter(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is a letter or digit.
 * <p>
 * A character is considered to be a letter or digit if either
 * {@code Character.isLetter(char ch)} or
 * {@code Character.isDigit(char ch)} returns
 * {@code true} for the character.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isLetterOrDigit(int)} method.
 *
 * @param   ch   the character to be tested.
 * @return  {@code true} if the character is a letter or digit;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isDigit(char)
 * @see     java.lang.Character#isJavaIdentifierPart(char)
 * @see     java.lang.Character#isJavaLetter(char)
 * @see     java.lang.Character#isJavaLetterOrDigit(char)
 * @see     java.lang.Character#isLetter(char)
 * @see     java.lang.Character#isUnicodeIdentifierPart(char)
 * @since   1.0.2
 */

public static boolean isLetterOrDigit(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a letter or digit.
 * <p>
 * A character is considered to be a letter or digit if either
 * {@link #isLetter(int) isLetter(codePoint)} or
 * {@link #isDigit(int) isDigit(codePoint)} returns
 * {@code true} for the character.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a letter or digit;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isDigit(int)
 * @see     java.lang.Character#isJavaIdentifierPart(int)
 * @see     java.lang.Character#isLetter(int)
 * @see     java.lang.Character#isUnicodeIdentifierPart(int)
 * @since   1.5
 */

public static boolean isLetterOrDigit(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the given {@code char} value is a
 * <a href="http://www.unicode.org/glossary/#low_surrogate_code_unit">
 * Unicode low-surrogate code unit</a>
 * (also known as <i>trailing-surrogate code unit</i>).
 *
 * <p>Such values do not represent characters by themselves,
 * but are used in the representation of
 * <a href="#supplementary">supplementary characters</a>
 * in the UTF-16 encoding.
 *
 * @param  ch the {@code char} value to be tested.
 * @return {@code true} if the {@code char} value is between
 *         {@link #MIN_LOW_SURROGATE} and
 *         {@link #MAX_LOW_SURROGATE} inclusive;
 *         {@code false} otherwise.
 * @see    java.lang.Character#isHighSurrogate(char)
 * @since  1.5
 */

public static boolean isLowSurrogate(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is a lowercase character.
 * <p>
 * A character is lowercase if its general category type, provided
 * by {@code Character.getType(ch)}, is
 * {@code LOWERCASE_LETTER}, or it has contributory property
 * Other_Lowercase as defined by the Unicode Standard.
 * <p>
 * The following are examples of lowercase characters:
 * <blockquote><pre>
 * a b c d e f g h i j k l m n o p q r s t u v w x y z
 * '&#92;u00DF' '&#92;u00E0' '&#92;u00E1' '&#92;u00E2' '&#92;u00E3' '&#92;u00E4' '&#92;u00E5' '&#92;u00E6'
 * '&#92;u00E7' '&#92;u00E8' '&#92;u00E9' '&#92;u00EA' '&#92;u00EB' '&#92;u00EC' '&#92;u00ED' '&#92;u00EE'
 * '&#92;u00EF' '&#92;u00F0' '&#92;u00F1' '&#92;u00F2' '&#92;u00F3' '&#92;u00F4' '&#92;u00F5' '&#92;u00F6'
 * '&#92;u00F8' '&#92;u00F9' '&#92;u00FA' '&#92;u00FB' '&#92;u00FC' '&#92;u00FD' '&#92;u00FE' '&#92;u00FF'
 * </pre></blockquote>
 * <p> Many other Unicode characters are lowercase too.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isLowerCase(int)} method.
 *
 * @param   ch   the character to be tested.
 * @return  {@code true} if the character is lowercase;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isLowerCase(char)
 * @see     java.lang.Character#isTitleCase(char)
 * @see     java.lang.Character#toLowerCase(char)
 * @see     java.lang.Character#getType(char)
 */

public static boolean isLowerCase(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a
 * lowercase character.
 * <p>
 * A character is lowercase if its general category type, provided
 * by {@link java.lang.Character#getType getType(codePoint)}, is
 * {@code LOWERCASE_LETTER}, or it has contributory property
 * Other_Lowercase as defined by the Unicode Standard.
 * <p>
 * The following are examples of lowercase characters:
 * <blockquote><pre>
 * a b c d e f g h i j k l m n o p q r s t u v w x y z
 * '&#92;u00DF' '&#92;u00E0' '&#92;u00E1' '&#92;u00E2' '&#92;u00E3' '&#92;u00E4' '&#92;u00E5' '&#92;u00E6'
 * '&#92;u00E7' '&#92;u00E8' '&#92;u00E9' '&#92;u00EA' '&#92;u00EB' '&#92;u00EC' '&#92;u00ED' '&#92;u00EE'
 * '&#92;u00EF' '&#92;u00F0' '&#92;u00F1' '&#92;u00F2' '&#92;u00F3' '&#92;u00F4' '&#92;u00F5' '&#92;u00F6'
 * '&#92;u00F8' '&#92;u00F9' '&#92;u00FA' '&#92;u00FB' '&#92;u00FC' '&#92;u00FD' '&#92;u00FE' '&#92;u00FF'
 * </pre></blockquote>
 * <p> Many other Unicode characters are lowercase too.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is lowercase;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isLowerCase(int)
 * @see     java.lang.Character#isTitleCase(int)
 * @see     java.lang.Character#toLowerCase(int)
 * @see     java.lang.Character#getType(int)
 * @since   1.5
 */

public static boolean isLowerCase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines whether the character is mirrored according to the
 * Unicode specification.  Mirrored characters should have their
 * glyphs horizontally mirrored when displayed in text that is
 * right-to-left.  For example, {@code '\u005Cu0028'} LEFT
 * PARENTHESIS is semantically defined to be an <i>opening
 * parenthesis</i>.  This will appear as a "(" in text that is
 * left-to-right but as a ")" in text that is right-to-left.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isMirrored(int)} method.
 *
 * @param  ch {@code char} for which the mirrored property is requested
 * @return {@code true} if the char is mirrored, {@code false}
 *         if the {@code char} is not mirrored or is not defined.
 * @since 1.4
 */

public static boolean isMirrored(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines whether the specified character (Unicode code point)
 * is mirrored according to the Unicode specification.  Mirrored
 * characters should have their glyphs horizontally mirrored when
 * displayed in text that is right-to-left.  For example,
 * {@code '\u005Cu0028'} LEFT PARENTHESIS is semantically
 * defined to be an <i>opening parenthesis</i>.  This will appear
 * as a "(" in text that is left-to-right but as a ")" in text
 * that is right-to-left.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is mirrored, {@code false}
 *          if the character is not mirrored or is not defined.
 * @since   1.5
 */

public static boolean isMirrored(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is ISO-LATIN-1 white space.
 * This method returns {@code true} for the following five
 * characters only:
 * <table class="striped">
 * <caption style="display:none">truechars</caption>
 * <thead>
 * <tr><th scope="col">Character
 *     <th scope="col">Code
 *     <th scope="col">Name
 * </thead>
 * <tbody>
 * <tr><th scope="row">{@code '\t'}</th>            <td>{@code U+0009}</td>
 *     <td>{@code HORIZONTAL TABULATION}</td></tr>
 * <tr><th scope="row">{@code '\n'}</th>            <td>{@code U+000A}</td>
 *     <td>{@code NEW LINE}</td></tr>
 * <tr><th scope="row">{@code '\f'}</th>            <td>{@code U+000C}</td>
 *     <td>{@code FORM FEED}</td></tr>
 * <tr><th scope="row">{@code '\r'}</th>            <td>{@code U+000D}</td>
 *     <td>{@code CARRIAGE RETURN}</td></tr>
 * <tr><th scope="row">{@code ' '}</th>  <td>{@code U+0020}</td>
 *     <td>{@code SPACE}</td></tr>
 * </tbody>
 * </table>
 *
 * @param      ch   the character to be tested.
 * @return     {@code true} if the character is ISO-LATIN-1 white
 *             space; {@code false} otherwise.
 * @see        java.lang.Character#isSpaceChar(char)
 * @see        java.lang.Character#isWhitespace(char)
 * @deprecated Replaced by isWhitespace(char).
 */

@Deprecated
public static boolean isSpace(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is a Unicode space character.
 * A character is considered to be a space character if and only if
 * it is specified to be a space character by the Unicode Standard. This
 * method returns true if the character's general category type is any of
 * the following:
 * <ul>
 * <li> {@code SPACE_SEPARATOR}
 * <li> {@code LINE_SEPARATOR}
 * <li> {@code PARAGRAPH_SEPARATOR}
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isSpaceChar(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return  {@code true} if the character is a space character;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isWhitespace(char)
 * @since   1.1
 */

public static boolean isSpaceChar(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a
 * Unicode space character.  A character is considered to be a
 * space character if and only if it is specified to be a space
 * character by the Unicode Standard. This method returns true if
 * the character's general category type is any of the following:
 *
 * <ul>
 * <li> {@link #SPACE_SEPARATOR}
 * <li> {@link #LINE_SEPARATOR}
 * <li> {@link #PARAGRAPH_SEPARATOR}
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a space character;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isWhitespace(int)
 * @since   1.5
 */

public static boolean isSpaceChar(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines whether the specified character (Unicode code point)
 * is in the <a href="#supplementary">supplementary character</a> range.
 *
 * @param  codePoint the character (Unicode code point) to be tested
 * @return {@code true} if the specified code point is between
 *         {@link #MIN_SUPPLEMENTARY_CODE_POINT} and
 *         {@link #MAX_CODE_POINT} inclusive;
 *         {@code false} otherwise.
 * @since  1.5
 */

public static boolean isSupplementaryCodePoint(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the given {@code char} value is a Unicode
 * <i>surrogate code unit</i>.
 *
 * <p>Such values do not represent characters by themselves,
 * but are used in the representation of
 * <a href="#supplementary">supplementary characters</a>
 * in the UTF-16 encoding.
 *
 * <p>A char value is a surrogate code unit if and only if it is either
 * a {@linkplain #isLowSurrogate(char) low-surrogate code unit} or
 * a {@linkplain #isHighSurrogate(char) high-surrogate code unit}.
 *
 * @param  ch the {@code char} value to be tested.
 * @return {@code true} if the {@code char} value is between
 *         {@link #MIN_SURROGATE} and
 *         {@link #MAX_SURROGATE} inclusive;
 *         {@code false} otherwise.
 * @since  1.7
 */

public static boolean isSurrogate(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines whether the specified pair of {@code char}
 * values is a valid
 * <a href="http://www.unicode.org/glossary/#surrogate_pair">
 * Unicode surrogate pair</a>.
 *
 * <p>This method is equivalent to the expression:
 * <blockquote><pre>{@code
 * isHighSurrogate(high) && isLowSurrogate(low)
 * }</pre></blockquote>
 *
 * @param  high the high-surrogate code value to be tested
 * @param  low the low-surrogate code value to be tested
 * @return {@code true} if the specified high and
 * low-surrogate code values represent a valid surrogate pair;
 * {@code false} otherwise.
 * @since  1.5
 */

public static boolean isSurrogatePair(char high, char low) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is a titlecase character.
 * <p>
 * A character is a titlecase character if its general
 * category type, provided by {@code Character.getType(ch)},
 * is {@code TITLECASE_LETTER}.
 * <p>
 * Some characters look like pairs of Latin letters. For example, there
 * is an uppercase letter that looks like "LJ" and has a corresponding
 * lowercase letter that looks like "lj". A third form, which looks like "Lj",
 * is the appropriate form to use when rendering a word in lowercase
 * with initial capitals, as for a book title.
 * <p>
 * These are some of the Unicode characters for which this method returns
 * {@code true}:
 * <ul>
 * <li>{@code LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON}
 * <li>{@code LATIN CAPITAL LETTER L WITH SMALL LETTER J}
 * <li>{@code LATIN CAPITAL LETTER N WITH SMALL LETTER J}
 * <li>{@code LATIN CAPITAL LETTER D WITH SMALL LETTER Z}
 * </ul>
 * <p> Many other Unicode characters are titlecase too.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isTitleCase(int)} method.
 *
 * @param   ch   the character to be tested.
 * @return  {@code true} if the character is titlecase;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isLowerCase(char)
 * @see     java.lang.Character#isUpperCase(char)
 * @see     java.lang.Character#toTitleCase(char)
 * @see     java.lang.Character#getType(char)
 * @since   1.0.2
 */

public static boolean isTitleCase(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is a titlecase character.
 * <p>
 * A character is a titlecase character if its general
 * category type, provided by {@link java.lang.Character#getType(int) getType(codePoint)},
 * is {@code TITLECASE_LETTER}.
 * <p>
 * Some characters look like pairs of Latin letters. For example, there
 * is an uppercase letter that looks like "LJ" and has a corresponding
 * lowercase letter that looks like "lj". A third form, which looks like "Lj",
 * is the appropriate form to use when rendering a word in lowercase
 * with initial capitals, as for a book title.
 * <p>
 * These are some of the Unicode characters for which this method returns
 * {@code true}:
 * <ul>
 * <li>{@code LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON}
 * <li>{@code LATIN CAPITAL LETTER L WITH SMALL LETTER J}
 * <li>{@code LATIN CAPITAL LETTER N WITH SMALL LETTER J}
 * <li>{@code LATIN CAPITAL LETTER D WITH SMALL LETTER Z}
 * </ul>
 * <p> Many other Unicode characters are titlecase too.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is titlecase;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isLowerCase(int)
 * @see     java.lang.Character#isUpperCase(int)
 * @see     java.lang.Character#toTitleCase(int)
 * @see     java.lang.Character#getType(int)
 * @since   1.5
 */

public static boolean isTitleCase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character may be part of a Unicode
 * identifier as other than the first character.
 * <p>
 * A character may be part of a Unicode identifier if and only if
 * one of the following statements is true:
 * <ul>
 * <li>  it is a letter
 * <li>  it is a connecting punctuation character (such as {@code '_'})
 * <li>  it is a digit
 * <li>  it is a numeric letter (such as a Roman numeral character)
 * <li>  it is a combining mark
 * <li>  it is a non-spacing mark
 * <li> {@code isIdentifierIgnorable} returns
 * {@code true} for this character.
 * <li> it is an <a href="http://www.unicode.org/reports/tr44/#Other_ID_Start">
 *      {@code Other_ID_Start}</a> character.
 * <li> it is an <a href="http://www.unicode.org/reports/tr44/#Other_ID_Continue">
 *      {@code Other_ID_Continue}</a> character.
 * </ul>
 * <p>
 * This method conforms to <a href="https://unicode.org/reports/tr31/#R1">
 * UAX31-R1: Default Identifiers</a> requirement of the Unicode Standard,
 * with the following profile of UAX31:
 * <pre>
 * Continue := Start + ID_Continue + ignorable
 * Medial := empty
 * ignorable := isIdentifierIgnorable(char) returns true for the character
 * </pre>
 * {@code ignorable} is added to {@code Continue} for backward
 * compatibility.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isUnicodeIdentifierPart(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return  {@code true} if the character may be part of a
 *          Unicode identifier; {@code false} otherwise.
 *
 * @spec https://www.unicode.org/reports/tr44 Unicode Character Database
 * @spec https://www.unicode.org/reports/tr31 Unicode Identifier and Pattern Syntax
 * @see     java.lang.Character#isIdentifierIgnorable(char)
 * @see     java.lang.Character#isJavaIdentifierPart(char)
 * @see     java.lang.Character#isLetterOrDigit(char)
 * @see     java.lang.Character#isUnicodeIdentifierStart(char)
 * @since   1.1
 */

public static boolean isUnicodeIdentifierPart(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) may be part of a Unicode
 * identifier as other than the first character.
 * <p>
 * A character may be part of a Unicode identifier if and only if
 * one of the following statements is true:
 * <ul>
 * <li>  it is a letter
 * <li>  it is a connecting punctuation character (such as {@code '_'})
 * <li>  it is a digit
 * <li>  it is a numeric letter (such as a Roman numeral character)
 * <li>  it is a combining mark
 * <li>  it is a non-spacing mark
 * <li> {@code isIdentifierIgnorable} returns
 * {@code true} for this character.
 * <li> it is an <a href="http://www.unicode.org/reports/tr44/#Other_ID_Start">
 *      {@code Other_ID_Start}</a> character.
 * <li> it is an <a href="http://www.unicode.org/reports/tr44/#Other_ID_Continue">
 *      {@code Other_ID_Continue}</a> character.
 * </ul>
 * <p>
 * This method conforms to <a href="https://unicode.org/reports/tr31/#R1">
 * UAX31-R1: Default Identifiers</a> requirement of the Unicode Standard,
 * with the following profile of UAX31:
 * <pre>
 * Continue := Start + ID_Continue + ignorable
 * Medial := empty
 * ignorable := isIdentifierIgnorable(int) returns true for the character
 * </pre>
 * {@code ignorable} is added to {@code Continue} for backward
 * compatibility.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character may be part of a
 *          Unicode identifier; {@code false} otherwise.
 *
 * @spec https://www.unicode.org/reports/tr44 Unicode Character Database
 * @spec https://www.unicode.org/reports/tr31 Unicode Identifier and Pattern Syntax
 * @see     java.lang.Character#isIdentifierIgnorable(int)
 * @see     java.lang.Character#isJavaIdentifierPart(int)
 * @see     java.lang.Character#isLetterOrDigit(int)
 * @see     java.lang.Character#isUnicodeIdentifierStart(int)
 * @since   1.5
 */

public static boolean isUnicodeIdentifierPart(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is permissible as the
 * first character in a Unicode identifier.
 * <p>
 * A character may start a Unicode identifier if and only if
 * one of the following conditions is true:
 * <ul>
 * <li> {@link #isLetter(char) isLetter(ch)} returns {@code true}
 * <li> {@link #getType(char) getType(ch)} returns
 *      {@code LETTER_NUMBER}.
 * <li> it is an <a href="http://www.unicode.org/reports/tr44/#Other_ID_Start">
 *      {@code Other_ID_Start}</a> character.
 * </ul>
 * <p>
 * This method conforms to <a href="https://unicode.org/reports/tr31/#R1">
 * UAX31-R1: Default Identifiers</a> requirement of the Unicode Standard,
 * with the following profile of UAX31:
 * <pre>
 * Start := ID_Start + 'VERTICAL TILDE' (U+2E2F)
 * </pre>
 * {@code 'VERTICAL TILDE'} is added to {@code Start} for backward
 * compatibility.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isUnicodeIdentifierStart(int)} method.
 *
 * @param   ch      the character to be tested.
 * @return  {@code true} if the character may start a Unicode
 *          identifier; {@code false} otherwise.
 *
 * @spec https://www.unicode.org/reports/tr44 Unicode Character Database
 * @spec https://www.unicode.org/reports/tr31 Unicode Identifier and Pattern Syntax
 * @see     java.lang.Character#isJavaIdentifierStart(char)
 * @see     java.lang.Character#isLetter(char)
 * @see     java.lang.Character#isUnicodeIdentifierPart(char)
 * @since   1.1
 */

public static boolean isUnicodeIdentifierStart(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is permissible as the
 * first character in a Unicode identifier.
 * <p>
 * A character may start a Unicode identifier if and only if
 * one of the following conditions is true:
 * <ul>
 * <li> {@link #isLetter(int) isLetter(codePoint)}
 *      returns {@code true}
 * <li> {@link #getType(int) getType(codePoint)}
 *      returns {@code LETTER_NUMBER}.
 * <li> it is an <a href="http://www.unicode.org/reports/tr44/#Other_ID_Start">
 *      {@code Other_ID_Start}</a> character.
 * </ul>
 * <p>
 * This method conforms to <a href="https://unicode.org/reports/tr31/#R1">
 * UAX31-R1: Default Identifiers</a> requirement of the Unicode Standard,
 * with the following profile of UAX31:
 * <pre>
 * Start := ID_Start + 'VERTICAL TILDE' (U+2E2F)
 * </pre>
 * {@code 'VERTICAL TILDE'} is added to {@code Start} for backward
 * compatibility.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character may start a Unicode
 *          identifier; {@code false} otherwise.
 *
 * @spec https://www.unicode.org/reports/tr44 Unicode Character Database
 * @spec https://www.unicode.org/reports/tr31 Unicode Identifier and Pattern Syntax
 * @see     java.lang.Character#isJavaIdentifierStart(int)
 * @see     java.lang.Character#isLetter(int)
 * @see     java.lang.Character#isUnicodeIdentifierPart(int)
 * @since   1.5
 */

public static boolean isUnicodeIdentifierStart(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is an uppercase character.
 * <p>
 * A character is uppercase if its general category type, provided by
 * {@code Character.getType(ch)}, is {@code UPPERCASE_LETTER}.
 * or it has contributory property Other_Uppercase as defined by the Unicode Standard.
 * <p>
 * The following are examples of uppercase characters:
 * <blockquote><pre>
 * A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
 * '&#92;u00C0' '&#92;u00C1' '&#92;u00C2' '&#92;u00C3' '&#92;u00C4' '&#92;u00C5' '&#92;u00C6' '&#92;u00C7'
 * '&#92;u00C8' '&#92;u00C9' '&#92;u00CA' '&#92;u00CB' '&#92;u00CC' '&#92;u00CD' '&#92;u00CE' '&#92;u00CF'
 * '&#92;u00D0' '&#92;u00D1' '&#92;u00D2' '&#92;u00D3' '&#92;u00D4' '&#92;u00D5' '&#92;u00D6' '&#92;u00D8'
 * '&#92;u00D9' '&#92;u00DA' '&#92;u00DB' '&#92;u00DC' '&#92;u00DD' '&#92;u00DE'
 * </pre></blockquote>
 * <p> Many other Unicode characters are uppercase too.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isUpperCase(int)} method.
 *
 * @param   ch   the character to be tested.
 * @return  {@code true} if the character is uppercase;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isLowerCase(char)
 * @see     java.lang.Character#isTitleCase(char)
 * @see     java.lang.Character#toUpperCase(char)
 * @see     java.lang.Character#getType(char)
 * @since   1.0
 */

public static boolean isUpperCase(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is an uppercase character.
 * <p>
 * A character is uppercase if its general category type, provided by
 * {@link java.lang.Character#getType(int) getType(codePoint)}, is {@code UPPERCASE_LETTER},
 * or it has contributory property Other_Uppercase as defined by the Unicode Standard.
 * <p>
 * The following are examples of uppercase characters:
 * <blockquote><pre>
 * A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
 * '&#92;u00C0' '&#92;u00C1' '&#92;u00C2' '&#92;u00C3' '&#92;u00C4' '&#92;u00C5' '&#92;u00C6' '&#92;u00C7'
 * '&#92;u00C8' '&#92;u00C9' '&#92;u00CA' '&#92;u00CB' '&#92;u00CC' '&#92;u00CD' '&#92;u00CE' '&#92;u00CF'
 * '&#92;u00D0' '&#92;u00D1' '&#92;u00D2' '&#92;u00D3' '&#92;u00D4' '&#92;u00D5' '&#92;u00D6' '&#92;u00D8'
 * '&#92;u00D9' '&#92;u00DA' '&#92;u00DB' '&#92;u00DC' '&#92;u00DD' '&#92;u00DE'
 * </pre></blockquote>
 * <p> Many other Unicode characters are uppercase too.
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is uppercase;
 *          {@code false} otherwise.
 * @see     java.lang.Character#isLowerCase(int)
 * @see     java.lang.Character#isTitleCase(int)
 * @see     java.lang.Character#toUpperCase(int)
 * @see     java.lang.Character#getType(int)
 * @since   1.5
 */

public static boolean isUpperCase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines whether the specified code point is a valid
 * <a href="http://www.unicode.org/glossary/#code_point">
 * Unicode code point value</a>.
 *
 * @param  codePoint the Unicode code point to be tested
 * @return {@code true} if the specified code point value is between
 *         {@link #MIN_CODE_POINT} and
 *         {@link #MAX_CODE_POINT} inclusive;
 *         {@code false} otherwise.
 * @since  1.5
 */

public static boolean isValidCodePoint(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character is white space according to Java.
 * A character is a Java whitespace character if and only if it satisfies
 * one of the following criteria:
 * <ul>
 * <li> It is a Unicode space character ({@code SPACE_SEPARATOR},
 *      {@code LINE_SEPARATOR}, or {@code PARAGRAPH_SEPARATOR})
 *      but is not also a non-breaking space ({@code '\u005Cu00A0'},
 *      {@code '\u005Cu2007'}, {@code '\u005Cu202F'}).
 * <li> It is {@code '\u005Ct'}, U+0009 HORIZONTAL TABULATION.
 * <li> It is {@code '\u005Cn'}, U+000A LINE FEED.
 * <li> It is {@code '\u005Cu000B'}, U+000B VERTICAL TABULATION.
 * <li> It is {@code '\u005Cf'}, U+000C FORM FEED.
 * <li> It is {@code '\u005Cr'}, U+000D CARRIAGE RETURN.
 * <li> It is {@code '\u005Cu001C'}, U+001C FILE SEPARATOR.
 * <li> It is {@code '\u005Cu001D'}, U+001D GROUP SEPARATOR.
 * <li> It is {@code '\u005Cu001E'}, U+001E RECORD SEPARATOR.
 * <li> It is {@code '\u005Cu001F'}, U+001F UNIT SEPARATOR.
 * </ul>
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #isWhitespace(int)} method.
 *
 * @param   ch the character to be tested.
 * @return  {@code true} if the character is a Java whitespace
 *          character; {@code false} otherwise.
 * @see     java.lang.Character#isSpaceChar(char)
 * @since   1.1
 */

public static boolean isWhitespace(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Determines if the specified character (Unicode code point) is
 * white space according to Java.  A character is a Java
 * whitespace character if and only if it satisfies one of the
 * following criteria:
 * <ul>
 * <li> It is a Unicode space character ({@link #SPACE_SEPARATOR},
 *      {@link #LINE_SEPARATOR}, or {@link #PARAGRAPH_SEPARATOR})
 *      but is not also a non-breaking space ({@code '\u005Cu00A0'},
 *      {@code '\u005Cu2007'}, {@code '\u005Cu202F'}).
 * <li> It is {@code '\u005Ct'}, U+0009 HORIZONTAL TABULATION.
 * <li> It is {@code '\u005Cn'}, U+000A LINE FEED.
 * <li> It is {@code '\u005Cu000B'}, U+000B VERTICAL TABULATION.
 * <li> It is {@code '\u005Cf'}, U+000C FORM FEED.
 * <li> It is {@code '\u005Cr'}, U+000D CARRIAGE RETURN.
 * <li> It is {@code '\u005Cu001C'}, U+001C FILE SEPARATOR.
 * <li> It is {@code '\u005Cu001D'}, U+001D GROUP SEPARATOR.
 * <li> It is {@code '\u005Cu001E'}, U+001E RECORD SEPARATOR.
 * <li> It is {@code '\u005Cu001F'}, U+001F UNIT SEPARATOR.
 * </ul>
 *
 * @param   codePoint the character (Unicode code point) to be tested.
 * @return  {@code true} if the character is a Java whitespace
 *          character; {@code false} otherwise.
 * @see     java.lang.Character#isSpaceChar(int)
 * @since   1.5
 */

public static boolean isWhitespace(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns the trailing surrogate (a
 * <a href="http://www.unicode.org/glossary/#low_surrogate_code_unit">
 * low surrogate code unit</a>) of the
 * <a href="http://www.unicode.org/glossary/#surrogate_pair">
 * surrogate pair</a>
 * representing the specified supplementary character (Unicode
 * code point) in the UTF-16 encoding.  If the specified character
 * is not a
 * <a href="Character.html#supplementary">supplementary character</a>,
 * an unspecified {@code char} is returned.
 *
 * <p>If
 * {@link #isSupplementaryCodePoint isSupplementaryCodePoint(x)}
 * is {@code true}, then
 * {@link #isLowSurrogate isLowSurrogate}{@code (lowSurrogate(x))} and
 * {@link #toCodePoint toCodePoint}{@code (}{@link #highSurrogate highSurrogate}{@code (x), lowSurrogate(x)) == x}
 * are also always {@code true}.
 *
 * @param   codePoint a supplementary character (Unicode code point)
 * @return  the trailing surrogate code unit used to represent the
 *          character in the UTF-16 encoding
 * @since   1.7
 */

public static char lowSurrogate(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns the index within the given {@code char} subarray
 * that is offset from the given {@code index} by
 * {@code codePointOffset} code points. The
 * {@code start} and {@code count} arguments specify a
 * subarray of the {@code char} array. Unpaired surrogates
 * within the text range given by {@code index} and
 * {@code codePointOffset} count as one code point each.
 *
 * @param a the {@code char} array
 * @param start the index of the first {@code char} of the
 * subarray
 * @param count the length of the subarray in {@code char}s
 * @param index the index to be offset
 * @param codePointOffset the offset in code points
 * @return the index within the subarray
 * @throws java.lang.NullPointerException if {@code a} is null.
 * @throws java.lang.IndexOutOfBoundsException
 *   if {@code start} or {@code count} is negative,
 *   or if {@code start + count} is larger than the length of
 *   the given array,
 *   or if {@code index} is less than {@code start} or
 *   larger then {@code start + count},
 *   or if {@code codePointOffset} is positive and the text range
 *   starting with {@code index} and ending with {@code start + count - 1}
 *   has fewer than {@code codePointOffset} code
 *   points,
 *   or if {@code codePointOffset} is negative and the text range
 *   starting with {@code start} and ending with {@code index - 1}
 *   has fewer than the absolute value of
 *   {@code codePointOffset} code points.
 * @since 1.5
 */

public static int offsetByCodePoints(char[] a, int start, int count, int index, int codePointOffset) { throw new RuntimeException("Stub!"); }

/**
 * Returns the index within the given char sequence that is offset
 * from the given {@code index} by {@code codePointOffset}
 * code points. Unpaired surrogates within the text range given by
 * {@code index} and {@code codePointOffset} count as
 * one code point each.
 *
 * @param seq the char sequence
 * @param index the index to be offset
 * @param codePointOffset the offset in code points
 * @return the index within the char sequence
 * @throws java.lang.NullPointerException if {@code seq} is null.
 * @throws java.lang.IndexOutOfBoundsException if {@code index}
 *   is negative or larger than the length of the char sequence,
 *   or if {@code codePointOffset} is positive and the
 *   subsequence starting with {@code index} has fewer than
 *   {@code codePointOffset} code points, or if
 *   {@code codePointOffset} is negative and the subsequence
 *   before {@code index} has fewer than the absolute value
 *   of {@code codePointOffset} code points.
 * @since 1.5
 */

public static int offsetByCodePoints(@androidx.annotation.RecentlyNonNull java.lang.CharSequence seq, int index, int codePointOffset) { throw new RuntimeException("Stub!"); }

/**
 * Returns the value obtained by reversing the order of the bytes in the
 * specified {@code char} value.
 *
 * @param ch The {@code char} of which to reverse the byte order.
 * @return the value obtained by reversing (or, equivalently, swapping)
 *     the bytes in the specified {@code char} value.
 * @since 1.5
 */

public static char reverseBytes(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Converts the specified character (Unicode code point) to its
 * UTF-16 representation stored in a {@code char} array. If
 * the specified code point is a BMP (Basic Multilingual Plane or
 * Plane 0) value, the resulting {@code char} array has
 * the same value as {@code codePoint}. If the specified code
 * point is a supplementary code point, the resulting
 * {@code char} array has the corresponding surrogate pair.
 *
 * @param  codePoint a Unicode code point
 * @return a {@code char} array having
 *         {@code codePoint}'s UTF-16 representation.
 * @throws java.lang.IllegalArgumentException if the specified
 * {@code codePoint} is not a valid Unicode code point.
 * @since  1.5
 */

public static char[] toChars(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Converts the specified character (Unicode code point) to its
 * UTF-16 representation. If the specified code point is a BMP
 * (Basic Multilingual Plane or Plane 0) value, the same value is
 * stored in {@code dst[dstIndex]}, and 1 is returned. If the
 * specified code point is a supplementary character, its
 * surrogate values are stored in {@code dst[dstIndex]}
 * (high-surrogate) and {@code dst[dstIndex+1]}
 * (low-surrogate), and 2 is returned.
 *
 * @param  codePoint the character (Unicode code point) to be converted.
 * @param  dst an array of {@code char} in which the
 * {@code codePoint}'s UTF-16 value is stored.
 * @param dstIndex the start index into the {@code dst}
 * array where the converted value is stored.
 * @return 1 if the code point is a BMP code point, 2 if the
 * code point is a supplementary code point.
 * @throws java.lang.IllegalArgumentException if the specified
 * {@code codePoint} is not a valid Unicode code point.
 * @throws java.lang.NullPointerException if the specified {@code dst} is null.
 * @throws java.lang.IndexOutOfBoundsException if {@code dstIndex}
 * is negative or not less than {@code dst.length}, or if
 * {@code dst} at {@code dstIndex} doesn't have enough
 * array element(s) to store the resulting {@code char}
 * value(s). (If {@code dstIndex} is equal to
 * {@code dst.length-1} and the specified
 * {@code codePoint} is a supplementary character, the
 * high-surrogate value is not stored in
 * {@code dst[dstIndex]}.)
 * @since  1.5
 */

public static int toChars(int codePoint, char[] dst, int dstIndex) { throw new RuntimeException("Stub!"); }

/**
 * Converts the specified surrogate pair to its supplementary code
 * point value. This method does not validate the specified
 * surrogate pair. The caller must validate it using {@link
 * #isSurrogatePair(char,char) isSurrogatePair} if necessary.
 *
 * @param  high the high-surrogate code unit
 * @param  low the low-surrogate code unit
 * @return the supplementary code point composed from the
 *         specified surrogate pair.
 * @since  1.5
 */

public static int toCodePoint(char high, char low) { throw new RuntimeException("Stub!"); }

/**
 * Converts the character argument to lowercase using case
 * mapping information from the UnicodeData file.
 * <p>
 * Note that
 * {@code Character.isLowerCase(Character.toLowerCase(ch))}
 * does not always return {@code true} for some ranges of
 * characters, particularly those that are symbols or ideographs.
 *
 * <p>In general, {@link java.lang.String#toLowerCase() String.toLowerCase()} should be used to map
 * characters to lowercase. {@code String} case mapping methods
 * have several benefits over {@code Character} case mapping methods.
 * {@code String} case mapping methods can perform locale-sensitive
 * mappings, context-sensitive mappings, and 1:M character mappings, whereas
 * the {@code Character} case mapping methods cannot.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #toLowerCase(int)} method.
 *
 * @param   ch   the character to be converted.
 * @return  the lowercase equivalent of the character, if any;
 *          otherwise, the character itself.
 * @see     java.lang.Character#isLowerCase(char)
 * @see     java.lang.String#toLowerCase()
 */

public static char toLowerCase(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Converts the character (Unicode code point) argument to
 * lowercase using case mapping information from the UnicodeData
 * file.
 *
 * <p> Note that
 * {@code Character.isLowerCase(Character.toLowerCase(codePoint))}
 * does not always return {@code true} for some ranges of
 * characters, particularly those that are symbols or ideographs.
 *
 * <p>In general, {@link java.lang.String#toLowerCase() String.toLowerCase()} should be used to map
 * characters to lowercase. {@code String} case mapping methods
 * have several benefits over {@code Character} case mapping methods.
 * {@code String} case mapping methods can perform locale-sensitive
 * mappings, context-sensitive mappings, and 1:M character mappings, whereas
 * the {@code Character} case mapping methods cannot.
 *
 * @param   codePoint   the character (Unicode code point) to be converted.
 * @return  the lowercase equivalent of the character (Unicode code
 *          point), if any; otherwise, the character itself.
 * @see     java.lang.Character#isLowerCase(int)
 * @see     java.lang.String#toLowerCase()
 *
 * @since   1.5
 */

public static int toLowerCase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code String} object representing this
 * {@code Character}'s value.  The result is a string of
 * length 1 whose sole component is the primitive
 * {@code char} value represented by this
 * {@code Character} object.
 *
 * @return  a string representation of this object.
 */

@androidx.annotation.RecentlyNonNull
public java.lang.String toString() { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code String} object representing the
 * specified {@code char}.  The result is a string of length
 * 1 consisting solely of the specified {@code char}.
 *
 * @param c the {@code char} to be converted
 * @return the string representation of the specified {@code char}
 * @since 1.4
 */

@androidx.annotation.RecentlyNonNull
public static java.lang.String toString(char c) { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code String} object representing the
 * specified character (Unicode code point).  The result is a string of
 * length 1 or 2, consisting solely of the specified {@code codePoint}.
 *
 * @param codePoint the {@code codePoint} to be converted
 * @return the string representation of the specified {@code codePoint}
 * @throws java.lang.IllegalArgumentException if the specified
 *      {@code codePoint} is not a {@linkplain #isValidCodePoint
 *      valid Unicode code point}.
 * @since 11
 */

@androidx.annotation.RecentlyNonNull
public static java.lang.String toString(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Converts the character argument to titlecase using case mapping
 * information from the UnicodeData file. If a character has no
 * explicit titlecase mapping and is not itself a titlecase char
 * according to UnicodeData, then the uppercase mapping is
 * returned as an equivalent titlecase mapping. If the
 * {@code char} argument is already a titlecase
 * {@code char}, the same {@code char} value will be
 * returned.
 * <p>
 * Note that
 * {@code Character.isTitleCase(Character.toTitleCase(ch))}
 * does not always return {@code true} for some ranges of
 * characters.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #toTitleCase(int)} method.
 *
 * @param   ch   the character to be converted.
 * @return  the titlecase equivalent of the character, if any;
 *          otherwise, the character itself.
 * @see     java.lang.Character#isTitleCase(char)
 * @see     java.lang.Character#toLowerCase(char)
 * @see     java.lang.Character#toUpperCase(char)
 * @since   1.0.2
 */

public static char toTitleCase(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Converts the character (Unicode code point) argument to titlecase using case mapping
 * information from the UnicodeData file. If a character has no
 * explicit titlecase mapping and is not itself a titlecase char
 * according to UnicodeData, then the uppercase mapping is
 * returned as an equivalent titlecase mapping. If the
 * character argument is already a titlecase
 * character, the same character value will be
 * returned.
 *
 * <p>Note that
 * {@code Character.isTitleCase(Character.toTitleCase(codePoint))}
 * does not always return {@code true} for some ranges of
 * characters.
 *
 * @param   codePoint   the character (Unicode code point) to be converted.
 * @return  the titlecase equivalent of the character, if any;
 *          otherwise, the character itself.
 * @see     java.lang.Character#isTitleCase(int)
 * @see     java.lang.Character#toLowerCase(int)
 * @see     java.lang.Character#toUpperCase(int)
 * @since   1.5
 */

public static int toTitleCase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Converts the character argument to uppercase using case mapping
 * information from the UnicodeData file.
 * <p>
 * Note that
 * {@code Character.isUpperCase(Character.toUpperCase(ch))}
 * does not always return {@code true} for some ranges of
 * characters, particularly those that are symbols or ideographs.
 *
 * <p>In general, {@link java.lang.String#toUpperCase() String.toUpperCase()} should be used to map
 * characters to uppercase. {@code String} case mapping methods
 * have several benefits over {@code Character} case mapping methods.
 * {@code String} case mapping methods can perform locale-sensitive
 * mappings, context-sensitive mappings, and 1:M character mappings, whereas
 * the {@code Character} case mapping methods cannot.
 *
 * <p><b>Note:</b> This method cannot handle <a
 * href="#supplementary"> supplementary characters</a>. To support
 * all Unicode characters, including supplementary characters, use
 * the {@link #toUpperCase(int)} method.
 *
 * @param   ch   the character to be converted.
 * @return  the uppercase equivalent of the character, if any;
 *          otherwise, the character itself.
 * @see     java.lang.Character#isUpperCase(char)
 * @see     java.lang.String#toUpperCase()
 */

public static char toUpperCase(char ch) { throw new RuntimeException("Stub!"); }

/**
 * Converts the character (Unicode code point) argument to
 * uppercase using case mapping information from the UnicodeData
 * file.
 *
 * <p>Note that
 * {@code Character.isUpperCase(Character.toUpperCase(codePoint))}
 * does not always return {@code true} for some ranges of
 * characters, particularly those that are symbols or ideographs.
 *
 * <p>In general, {@link java.lang.String#toUpperCase() String.toUpperCase()} should be used to map
 * characters to uppercase. {@code String} case mapping methods
 * have several benefits over {@code Character} case mapping methods.
 * {@code String} case mapping methods can perform locale-sensitive
 * mappings, context-sensitive mappings, and 1:M character mappings, whereas
 * the {@code Character} case mapping methods cannot.
 *
 * @param   codePoint   the character (Unicode code point) to be converted.
 * @return  the uppercase equivalent of the character, if any;
 *          otherwise, the character itself.
 * @see     java.lang.Character#isUpperCase(int)
 * @see     java.lang.String#toUpperCase()
 *
 * @since   1.5
 */

public static int toUpperCase(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@code Character} instance representing the specified
 * {@code char} value.
 * If a new {@code Character} instance is not required, this method
 * should generally be used in preference to the constructor
 * {@link #Character(char)}, as this method is likely to yield
 * significantly better space and time performance by caching
 * frequently requested values.
 *
 * This method will always cache values in the range {@code
 * '\u005Cu0000'} to {@code '\u005Cu007F'}, inclusive, and may
 * cache other values outside of this range.
 *
 * @param  c a char value.
 * @return a {@code Character} instance representing {@code c}.
 * @since  1.5
 */

@androidx.annotation.RecentlyNonNull
public static java.lang.Character valueOf(char c) { throw new RuntimeException("Stub!"); }

/**
 * The number of bytes used to represent a {@code char} value in unsigned
 * binary form.
 *
 * @since 1.8
 */

public static final int BYTES = 2; // 0x2

/**
 * General category "Mc" in the Unicode specification.
 * @since   1.1
 */

public static final byte COMBINING_SPACING_MARK = 8; // 0x8

/**
 * General category "Pc" in the Unicode specification.
 * @since   1.1
 */

public static final byte CONNECTOR_PUNCTUATION = 23; // 0x17

/**
 * General category "Cc" in the Unicode specification.
 * @since   1.1
 */

public static final byte CONTROL = 15; // 0xf

/**
 * General category "Sc" in the Unicode specification.
 * @since   1.1
 */

public static final byte CURRENCY_SYMBOL = 26; // 0x1a

/**
 * General category "Pd" in the Unicode specification.
 * @since   1.1
 */

public static final byte DASH_PUNCTUATION = 20; // 0x14

/**
 * General category "Nd" in the Unicode specification.
 * @since   1.1
 */

public static final byte DECIMAL_DIGIT_NUMBER = 9; // 0x9

/**
 * Weak bidirectional character type "AN" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_ARABIC_NUMBER = 6; // 0x6

/**
 * Weak bidirectional character type "BN" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_BOUNDARY_NEUTRAL = 9; // 0x9

/**
 * Weak bidirectional character type "CS" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_COMMON_NUMBER_SEPARATOR = 7; // 0x7

/**
 * Weak bidirectional character type "EN" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_EUROPEAN_NUMBER = 3; // 0x3

/**
 * Weak bidirectional character type "ES" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_EUROPEAN_NUMBER_SEPARATOR = 4; // 0x4

/**
 * Weak bidirectional character type "ET" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_EUROPEAN_NUMBER_TERMINATOR = 5; // 0x5

/**
 * Weak bidirectional character type "FSI" in the Unicode specification.
 * @since 9
 */

public static final byte DIRECTIONALITY_FIRST_STRONG_ISOLATE = 21; // 0x15

/**
 * Strong bidirectional character type "L" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_LEFT_TO_RIGHT = 0; // 0x0

/**
 * Strong bidirectional character type "LRE" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_LEFT_TO_RIGHT_EMBEDDING = 14; // 0xe

/**
 * Weak bidirectional character type "LRI" in the Unicode specification.
 * @since 9
 */

public static final byte DIRECTIONALITY_LEFT_TO_RIGHT_ISOLATE = 19; // 0x13

/**
 * Strong bidirectional character type "LRO" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_LEFT_TO_RIGHT_OVERRIDE = 15; // 0xf

/**
 * Weak bidirectional character type "NSM" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_NONSPACING_MARK = 8; // 0x8

/**
 * Neutral bidirectional character type "ON" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_OTHER_NEUTRALS = 13; // 0xd

/**
 * Neutral bidirectional character type "B" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_PARAGRAPH_SEPARATOR = 10; // 0xa

/**
 * Weak bidirectional character type "PDF" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_POP_DIRECTIONAL_FORMAT = 18; // 0x12

/**
 * Weak bidirectional character type "PDI" in the Unicode specification.
 * @since 9
 */

public static final byte DIRECTIONALITY_POP_DIRECTIONAL_ISOLATE = 22; // 0x16

/**
 * Strong bidirectional character type "R" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_RIGHT_TO_LEFT = 1; // 0x1

/**
 * Strong bidirectional character type "AL" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC = 2; // 0x2

/**
 * Strong bidirectional character type "RLE" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_EMBEDDING = 16; // 0x10

/**
 * Weak bidirectional character type "RLI" in the Unicode specification.
 * @since 9
 */

public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_ISOLATE = 20; // 0x14

/**
 * Strong bidirectional character type "RLO" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_OVERRIDE = 17; // 0x11

/**
 * Neutral bidirectional character type "S" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_SEGMENT_SEPARATOR = 11; // 0xb

/**
 * Undefined bidirectional character type. Undefined {@code char}
 * values have undefined directionality in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_UNDEFINED = -1; // 0xffffffff

/**
 * Neutral bidirectional character type "WS" in the Unicode specification.
 * @since 1.4
 */

public static final byte DIRECTIONALITY_WHITESPACE = 12; // 0xc

/**
 * General category "Me" in the Unicode specification.
 * @since   1.1
 */

public static final byte ENCLOSING_MARK = 7; // 0x7

/**
 * General category "Pe" in the Unicode specification.
 * @since   1.1
 */

public static final byte END_PUNCTUATION = 22; // 0x16

/**
 * General category "Pf" in the Unicode specification.
 * @since   1.4
 */

public static final byte FINAL_QUOTE_PUNCTUATION = 30; // 0x1e

/**
 * General category "Cf" in the Unicode specification.
 * @since   1.1
 */

public static final byte FORMAT = 16; // 0x10

/**
 * General category "Pi" in the Unicode specification.
 * @since   1.4
 */

public static final byte INITIAL_QUOTE_PUNCTUATION = 29; // 0x1d

/**
 * General category "Nl" in the Unicode specification.
 * @since   1.1
 */

public static final byte LETTER_NUMBER = 10; // 0xa

/**
 * General category "Zl" in the Unicode specification.
 * @since   1.1
 */

public static final byte LINE_SEPARATOR = 13; // 0xd

/**
 * General category "Ll" in the Unicode specification.
 * @since   1.1
 */

public static final byte LOWERCASE_LETTER = 2; // 0x2

/**
 * General category "Sm" in the Unicode specification.
 * @since   1.1
 */

public static final byte MATH_SYMBOL = 25; // 0x19

/**
 * The maximum value of a
 * <a href="http://www.unicode.org/glossary/#code_point">
 * Unicode code point</a>, constant {@code U+10FFFF}.
 *
 * @since 1.5
 */

public static final int MAX_CODE_POINT = 1114111; // 0x10ffff

/**
 * The maximum value of a
 * <a href="http://www.unicode.org/glossary/#high_surrogate_code_unit">
 * Unicode high-surrogate code unit</a>
 * in the UTF-16 encoding, constant {@code '\u005CuDBFF'}.
 * A high-surrogate is also known as a <i>leading-surrogate</i>.
 *
 * @since 1.5
 */

public static final char MAX_HIGH_SURROGATE = 56319; // 0xdbff '\udbff'

/**
 * The maximum value of a
 * <a href="http://www.unicode.org/glossary/#low_surrogate_code_unit">
 * Unicode low-surrogate code unit</a>
 * in the UTF-16 encoding, constant {@code '\u005CuDFFF'}.
 * A low-surrogate is also known as a <i>trailing-surrogate</i>.
 *
 * @since 1.5
 */

public static final char MAX_LOW_SURROGATE = 57343; // 0xdfff '\udfff'

/**
 * The maximum radix available for conversion to and from strings.
 * The constant value of this field is the largest value permitted
 * for the radix argument in radix-conversion methods such as the
 * {@code digit} method, the {@code forDigit} method, and the
 * {@code toString} method of class {@code Integer}.
 *
 * @see     java.lang.Character#digit(char, int)
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Integer#toString(int, int)
 * @see     java.lang.Integer#valueOf(String)
 */

public static final int MAX_RADIX = 36; // 0x24

/**
 * The maximum value of a Unicode surrogate code unit in the
 * UTF-16 encoding, constant {@code '\u005CuDFFF'}.
 *
 * @since 1.5
 */

public static final char MAX_SURROGATE = 57343; // 0xdfff '\udfff'

/**
 * The constant value of this field is the largest value of type
 * {@code char}, {@code '\u005CuFFFF'}.
 *
 * @since   1.0.2
 */

public static final char MAX_VALUE = 65535; // 0xffff '\uffff'

/**
 * The minimum value of a
 * <a href="http://www.unicode.org/glossary/#code_point">
 * Unicode code point</a>, constant {@code U+0000}.
 *
 * @since 1.5
 */

public static final int MIN_CODE_POINT = 0; // 0x0

/**
 * The minimum value of a
 * <a href="http://www.unicode.org/glossary/#high_surrogate_code_unit">
 * Unicode high-surrogate code unit</a>
 * in the UTF-16 encoding, constant {@code '\u005CuD800'}.
 * A high-surrogate is also known as a <i>leading-surrogate</i>.
 *
 * @since 1.5
 */

public static final char MIN_HIGH_SURROGATE = 55296; // 0xd800 '\ud800'

/**
 * The minimum value of a
 * <a href="http://www.unicode.org/glossary/#low_surrogate_code_unit">
 * Unicode low-surrogate code unit</a>
 * in the UTF-16 encoding, constant {@code '\u005CuDC00'}.
 * A low-surrogate is also known as a <i>trailing-surrogate</i>.
 *
 * @since 1.5
 */

public static final char MIN_LOW_SURROGATE = 56320; // 0xdc00 '\udc00'

/**
 * The minimum radix available for conversion to and from strings.
 * The constant value of this field is the smallest value permitted
 * for the radix argument in radix-conversion methods such as the
 * {@code digit} method, the {@code forDigit} method, and the
 * {@code toString} method of class {@code Integer}.
 *
 * @see     java.lang.Character#digit(char, int)
 * @see     java.lang.Character#forDigit(int, int)
 * @see     java.lang.Integer#toString(int, int)
 * @see     java.lang.Integer#valueOf(String)
 */

public static final int MIN_RADIX = 2; // 0x2

/**
 * The minimum value of a
 * <a href="http://www.unicode.org/glossary/#supplementary_code_point">
 * Unicode supplementary code point</a>, constant {@code U+10000}.
 *
 * @since 1.5
 */

public static final int MIN_SUPPLEMENTARY_CODE_POINT = 65536; // 0x10000

/**
 * The minimum value of a Unicode surrogate code unit in the
 * UTF-16 encoding, constant {@code '\u005CuD800'}.
 *
 * @since 1.5
 */

public static final char MIN_SURROGATE = 55296; // 0xd800 '\ud800'

/**
 * The constant value of this field is the smallest value of type
 * {@code char}, {@code '\u005Cu0000'}.
 *
 * @since   1.0.2
 */

public static final char MIN_VALUE = 0; // 0x0000 '\u0000'

/**
 * General category "Lm" in the Unicode specification.
 * @since   1.1
 */

public static final byte MODIFIER_LETTER = 4; // 0x4

/**
 * General category "Sk" in the Unicode specification.
 * @since   1.1
 */

public static final byte MODIFIER_SYMBOL = 27; // 0x1b

/**
 * General category "Mn" in the Unicode specification.
 * @since   1.1
 */

public static final byte NON_SPACING_MARK = 6; // 0x6

/**
 * General category "Lo" in the Unicode specification.
 * @since   1.1
 */

public static final byte OTHER_LETTER = 5; // 0x5

/**
 * General category "No" in the Unicode specification.
 * @since   1.1
 */

public static final byte OTHER_NUMBER = 11; // 0xb

/**
 * General category "Po" in the Unicode specification.
 * @since   1.1
 */

public static final byte OTHER_PUNCTUATION = 24; // 0x18

/**
 * General category "So" in the Unicode specification.
 * @since   1.1
 */

public static final byte OTHER_SYMBOL = 28; // 0x1c

/**
 * General category "Zp" in the Unicode specification.
 * @since   1.1
 */

public static final byte PARAGRAPH_SEPARATOR = 14; // 0xe

/**
 * General category "Co" in the Unicode specification.
 * @since   1.1
 */

public static final byte PRIVATE_USE = 18; // 0x12

/**
 * The number of bits used to represent a {@code char} value in unsigned
 * binary form, constant {@code 16}.
 *
 * @since 1.5
 */

public static final int SIZE = 16; // 0x10

/**
 * General category "Zs" in the Unicode specification.
 * @since   1.1
 */

public static final byte SPACE_SEPARATOR = 12; // 0xc

/**
 * General category "Ps" in the Unicode specification.
 * @since   1.1
 */

public static final byte START_PUNCTUATION = 21; // 0x15

/**
 * General category "Cs" in the Unicode specification.
 * @since   1.1
 */

public static final byte SURROGATE = 19; // 0x13

/**
 * General category "Lt" in the Unicode specification.
 * @since   1.1
 */

public static final byte TITLECASE_LETTER = 3; // 0x3

/**
 * The {@code Class} instance representing the primitive type
 * {@code char}.
 *
 * @since   1.1
 */

public static final java.lang.Class<java.lang.Character> TYPE;
static { TYPE = null; }

/**
 * General category "Cn" in the Unicode specification.
 * @since   1.1
 */

public static final byte UNASSIGNED = 0; // 0x0

/**
 * General category "Lu" in the Unicode specification.
 * @since   1.1
 */

public static final byte UPPERCASE_LETTER = 1; // 0x1
/**
 * Instances of this class represent particular subsets of the Unicode
 * character set.  The only family of subsets defined in the
 * {@code Character} class is {@link java.lang.Character.UnicodeBlock Character.UnicodeBlock}.
 * Other portions of the Java API may define other subsets for their
 * own purposes.
 *
 * @since 1.2
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class Subset {

/**
 * Constructs a new {@code Subset} instance.
 *
 * @param  name  The name of this subset
 * @throws java.lang.NullPointerException if name is {@code null}
 */

protected Subset(@androidx.annotation.RecentlyNonNull java.lang.String name) { throw new RuntimeException("Stub!"); }

/**
 * Compares two {@code Subset} objects for equality.
 * This method returns {@code true} if and only if
 * {@code this} and the argument refer to the same
 * object; since this method is {@code final}, this
 * guarantee holds for all subclasses.
 */

public final boolean equals(@androidx.annotation.RecentlyNullable java.lang.Object obj) { throw new RuntimeException("Stub!"); }

/**
 * Returns the standard hash code as defined by the
 * {@link java.lang.Object#hashCode Object.hashCode} method.  This method
 * is {@code final} in order to ensure that the
 * {@code equals} and {@code hashCode} methods will
 * be consistent in all subclasses.
 */

public final int hashCode() { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of this subset.
 */

@androidx.annotation.RecentlyNonNull
public final java.lang.String toString() { throw new RuntimeException("Stub!"); }
}

/**
 * A family of character subsets representing the character blocks in the
 * Unicode specification. Character blocks generally define characters
 * used for a specific script or purpose. A character is contained by
 * at most one Unicode block.
 *
 * @since 1.2
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class UnicodeBlock extends java.lang.Character.Subset {

UnicodeBlock() { super(""); throw new RuntimeException("Stub!"); }

/**
 * Returns the UnicodeBlock with the given name. Block
 * names are determined by The Unicode Standard. The file
 * {@code Blocks.txt} defines blocks for a particular
 * version of the standard. The {@link java.lang.Character Character} class specifies
 * the version of the standard that it supports.
 * <p>
 * This method accepts block names in the following forms:
 * <ol>
 * <li> Canonical block names as defined by the Unicode Standard.
 * For example, the standard defines a "Basic Latin" block. Therefore, this
 * method accepts "Basic Latin" as a valid block name. The documentation of
 * each UnicodeBlock provides the canonical name.
 * <li>Canonical block names with all spaces removed. For example, "BasicLatin"
 * is a valid block name for the "Basic Latin" block.
 * <li>The text representation of each constant UnicodeBlock identifier.
 * For example, this method will return the {@link #BASIC_LATIN} block if
 * provided with the "BASIC_LATIN" name. This form replaces all spaces and
 * hyphens in the canonical name with underscores.
 * </ol>
 * Finally, character case is ignored for all of the valid block name forms.
 * For example, "BASIC_LATIN" and "basic_latin" are both valid block names.
 * The en_US locale's case mapping rules are used to provide case-insensitive
 * string comparisons for block name validation.
 * <p>
 * If the Unicode Standard changes block names, both the previous and
 * current names will be accepted.
 *
 * @param blockName A {@code UnicodeBlock} name.
 * @return The {@code UnicodeBlock} instance identified
 *         by {@code blockName}
 * @throws java.lang.IllegalArgumentException if {@code blockName} is an
 *         invalid name
 * @throws java.lang.NullPointerException if {@code blockName} is null
 * @since 1.5
 */

@androidx.annotation.RecentlyNonNull
public static java.lang.Character.UnicodeBlock forName(@androidx.annotation.RecentlyNonNull java.lang.String blockName) { throw new RuntimeException("Stub!"); }

/**
 * Returns the object representing the Unicode block containing the
 * given character, or {@code null} if the character is not a
 * member of a defined block.
 *
 * <p><b>Note:</b> This method cannot handle
 * <a href="Character.html#supplementary"> supplementary
 * characters</a>.  To support all Unicode characters, including
 * supplementary characters, use the {@link #of(int)} method.
 *
 * @param   c  The character in question
 * @return  The {@code UnicodeBlock} instance representing the
 *          Unicode block of which this character is a member, or
 *          {@code null} if the character is not a member of any
 *          Unicode block
 */

@androidx.annotation.RecentlyNullable
public static java.lang.Character.UnicodeBlock of(char c) { throw new RuntimeException("Stub!"); }

/**
 * Returns the object representing the Unicode block
 * containing the given character (Unicode code point), or
 * {@code null} if the character is not a member of a
 * defined block.
 *
 * @param   codePoint the character (Unicode code point) in question.
 * @return  The {@code UnicodeBlock} instance representing the
 *          Unicode block of which this character is a member, or
 *          {@code null} if the character is not a member of any
 *          Unicode block
 * @throws  java.lang.IllegalArgumentException if the specified
 * {@code codePoint} is an invalid Unicode code point.
 * @see java.lang.Character#isValidCodePoint(int)
 * @since   1.5
 */

@androidx.annotation.RecentlyNullable
public static java.lang.Character.UnicodeBlock of(int codePoint) { throw new RuntimeException("Stub!"); }

/**
 * Constant for the "Adlam" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock ADLAM;
static { ADLAM = null; }

/**
 * Constant for the "Aegean Numbers" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock AEGEAN_NUMBERS;
static { AEGEAN_NUMBERS = null; }

/**
 * Constant for the "Ahom" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock AHOM;
static { AHOM = null; }

/**
 * Constant for the "Alchemical Symbols" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ALCHEMICAL_SYMBOLS;
static { ALCHEMICAL_SYMBOLS = null; }

/**
 * Constant for the "Alphabetic Presentation Forms" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ALPHABETIC_PRESENTATION_FORMS;
static { ALPHABETIC_PRESENTATION_FORMS = null; }

/**
 * Constant for the "Anatolian Hieroglyphs" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock ANATOLIAN_HIEROGLYPHS;
static { ANATOLIAN_HIEROGLYPHS = null; }

/**
 * Constant for the "Ancient Greek Musical Notation" Unicode character
 * block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ANCIENT_GREEK_MUSICAL_NOTATION;
static { ANCIENT_GREEK_MUSICAL_NOTATION = null; }

/**
 * Constant for the "Ancient Greek Numbers" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ANCIENT_GREEK_NUMBERS;
static { ANCIENT_GREEK_NUMBERS = null; }

/**
 * Constant for the "Ancient Symbols" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ANCIENT_SYMBOLS;
static { ANCIENT_SYMBOLS = null; }

/**
 * Constant for the "Arabic" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ARABIC;
static { ARABIC = null; }

/**
 * Constant for the "Arabic Extended-A" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock ARABIC_EXTENDED_A;
static { ARABIC_EXTENDED_A = null; }

/**
 * Constant for the "Arabic Extended-B" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock ARABIC_EXTENDED_B;
static { ARABIC_EXTENDED_B = null; }

/**
 * Constant for the "Arabic Extended-C" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock ARABIC_EXTENDED_C;
static { ARABIC_EXTENDED_C = null; }

/**
 * Constant for the "Arabic Mathematical Alphabetic Symbols" Unicode
 * character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock ARABIC_MATHEMATICAL_ALPHABETIC_SYMBOLS;
static { ARABIC_MATHEMATICAL_ALPHABETIC_SYMBOLS = null; }

/**
 * Constant for the "Arabic Presentation Forms-A" Unicode character
 * block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ARABIC_PRESENTATION_FORMS_A;
static { ARABIC_PRESENTATION_FORMS_A = null; }

/**
 * Constant for the "Arabic Presentation Forms-B" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ARABIC_PRESENTATION_FORMS_B;
static { ARABIC_PRESENTATION_FORMS_B = null; }

/**
 * Constant for the "Arabic Supplement" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ARABIC_SUPPLEMENT;
static { ARABIC_SUPPLEMENT = null; }

/**
 * Constant for the "Armenian" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ARMENIAN;
static { ARMENIAN = null; }

/**
 * Constant for the "Arrows" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ARROWS;
static { ARROWS = null; }

/**
 * Constant for the "Avestan" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock AVESTAN;
static { AVESTAN = null; }

/**
 * Constant for the "Balinese" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock BALINESE;
static { BALINESE = null; }

/**
 * Constant for the "Bamum" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock BAMUM;
static { BAMUM = null; }

/**
 * Constant for the "Bamum Supplement" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock BAMUM_SUPPLEMENT;
static { BAMUM_SUPPLEMENT = null; }

/**
 * Constant for the "Basic Latin" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock BASIC_LATIN;
static { BASIC_LATIN = null; }

/**
 * Constant for the "Bassa Vah" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock BASSA_VAH;
static { BASSA_VAH = null; }

/**
 * Constant for the "Batak" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock BATAK;
static { BATAK = null; }

/**
 * Constant for the "Bengali" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock BENGALI;
static { BENGALI = null; }

/**
 * Constant for the "Bhaiksuki" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock BHAIKSUKI;
static { BHAIKSUKI = null; }

/**
 * Constant for the "Block Elements" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock BLOCK_ELEMENTS;
static { BLOCK_ELEMENTS = null; }

/**
 * Constant for the "Bopomofo" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock BOPOMOFO;
static { BOPOMOFO = null; }

/**
 * Constant for the "Bopomofo Extended" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock BOPOMOFO_EXTENDED;
static { BOPOMOFO_EXTENDED = null; }

/**
 * Constant for the "Box Drawing" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock BOX_DRAWING;
static { BOX_DRAWING = null; }

/**
 * Constant for the "Brahmi" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock BRAHMI;
static { BRAHMI = null; }

/**
 * Constant for the "Braille Patterns" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock BRAILLE_PATTERNS;
static { BRAILLE_PATTERNS = null; }

/**
 * Constant for the "Buginese" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock BUGINESE;
static { BUGINESE = null; }

/**
 * Constant for the "Buhid" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock BUHID;
static { BUHID = null; }

/**
 * Constant for the "Byzantine Musical Symbols" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock BYZANTINE_MUSICAL_SYMBOLS;
static { BYZANTINE_MUSICAL_SYMBOLS = null; }

/**
 * Constant for the "Carian" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CARIAN;
static { CARIAN = null; }

/**
 * Constant for the "Caucasian Albanian" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock CAUCASIAN_ALBANIAN;
static { CAUCASIAN_ALBANIAN = null; }

/**
 * Constant for the "Chakma" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock CHAKMA;
static { CHAKMA = null; }

/**
 * Constant for the "Cham" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CHAM;
static { CHAM = null; }

/**
 * Constant for the "Cherokee" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock CHEROKEE;
static { CHEROKEE = null; }

/**
 * Constant for the "Cherokee Supplement" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock CHEROKEE_SUPPLEMENT;
static { CHEROKEE_SUPPLEMENT = null; }

/**
 * Constant for the "Chess Symbols" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock CHESS_SYMBOLS;
static { CHESS_SYMBOLS = null; }

/**
 * Constant for the "Chorasmian" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock CHORASMIAN;
static { CHORASMIAN = null; }

/**
 * Constant for the "CJK Compatibility" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CJK_COMPATIBILITY;
static { CJK_COMPATIBILITY = null; }

/**
 * Constant for the "CJK Compatibility Forms" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CJK_COMPATIBILITY_FORMS;
static { CJK_COMPATIBILITY_FORMS = null; }

/**
 * Constant for the "CJK Compatibility Ideographs" Unicode character
 * block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CJK_COMPATIBILITY_IDEOGRAPHS;
static { CJK_COMPATIBILITY_IDEOGRAPHS = null; }

/**
 * Constant for the "CJK Compatibility Ideographs Supplement" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT;
static { CJK_COMPATIBILITY_IDEOGRAPHS_SUPPLEMENT = null; }

/**
 * Constant for the "CJK Radicals Supplement" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock CJK_RADICALS_SUPPLEMENT;
static { CJK_RADICALS_SUPPLEMENT = null; }

/**
 * Constant for the "CJK Strokes" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CJK_STROKES;
static { CJK_STROKES = null; }

/**
 * Constant for the "CJK Symbols and Punctuation" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CJK_SYMBOLS_AND_PUNCTUATION;
static { CJK_SYMBOLS_AND_PUNCTUATION = null; }

/**
 * Constant for the "CJK Unified Ideographs" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS;
static { CJK_UNIFIED_IDEOGRAPHS = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension A" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension B" Unicode
 * character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension C" Unicode
 * character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension D" Unicode
 * character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_D;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_D = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension E" Unicode
 * character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_E;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_E = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension F" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_F;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_F = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension G" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_G;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_G = null; }

/**
 * Constant for the "CJK Unified Ideographs Extension H" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock CJK_UNIFIED_IDEOGRAPHS_EXTENSION_H;
static { CJK_UNIFIED_IDEOGRAPHS_EXTENSION_H = null; }

/**
 * Constant for the "Combining Diacritical Marks" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock COMBINING_DIACRITICAL_MARKS;
static { COMBINING_DIACRITICAL_MARKS = null; }

/**
 * Constant for the "Combining Diacritical Marks Extended" Unicode
 * character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock COMBINING_DIACRITICAL_MARKS_EXTENDED;
static { COMBINING_DIACRITICAL_MARKS_EXTENDED = null; }

/**
 * Constant for the "Combining Diacritical Marks Supplement" Unicode
 * character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock COMBINING_DIACRITICAL_MARKS_SUPPLEMENT;
static { COMBINING_DIACRITICAL_MARKS_SUPPLEMENT = null; }

/**
 * Constant for the "Combining Half Marks" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock COMBINING_HALF_MARKS;
static { COMBINING_HALF_MARKS = null; }

/**
 * Constant for the "Combining Diacritical Marks for Symbols" Unicode
 * character block.
 * <p>
 * This block was previously known as "Combining Marks for Symbols".
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock COMBINING_MARKS_FOR_SYMBOLS;
static { COMBINING_MARKS_FOR_SYMBOLS = null; }

/**
 * Constant for the "Common Indic Number Forms" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock COMMON_INDIC_NUMBER_FORMS;
static { COMMON_INDIC_NUMBER_FORMS = null; }

/**
 * Constant for the "Control Pictures" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CONTROL_PICTURES;
static { CONTROL_PICTURES = null; }

/**
 * Constant for the "Coptic" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock COPTIC;
static { COPTIC = null; }

/**
 * Constant for the "Coptic Epact Numbers" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock COPTIC_EPACT_NUMBERS;
static { COPTIC_EPACT_NUMBERS = null; }

/**
 * Constant for the "Counting Rod Numerals" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock COUNTING_ROD_NUMERALS;
static { COUNTING_ROD_NUMERALS = null; }

/**
 * Constant for the "Cuneiform" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CUNEIFORM;
static { CUNEIFORM = null; }

/**
 * Constant for the "Cuneiform Numbers and Punctuation" Unicode
 * character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CUNEIFORM_NUMBERS_AND_PUNCTUATION;
static { CUNEIFORM_NUMBERS_AND_PUNCTUATION = null; }

/**
 * Constant for the "Currency Symbols" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CURRENCY_SYMBOLS;
static { CURRENCY_SYMBOLS = null; }

/**
 * Constant for the "Cypriot Syllabary" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock CYPRIOT_SYLLABARY;
static { CYPRIOT_SYLLABARY = null; }

/**
 * Constant for the "Cypro-Minoan" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock CYPRO_MINOAN;
static { CYPRO_MINOAN = null; }

/**
 * Constant for the "Cyrillic" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock CYRILLIC;
static { CYRILLIC = null; }

/**
 * Constant for the "Cyrillic Extended-A" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CYRILLIC_EXTENDED_A;
static { CYRILLIC_EXTENDED_A = null; }

/**
 * Constant for the "Cyrillic Extended-B" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock CYRILLIC_EXTENDED_B;
static { CYRILLIC_EXTENDED_B = null; }

/**
 * Constant for the "Cyrillic Extended-C" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock CYRILLIC_EXTENDED_C;
static { CYRILLIC_EXTENDED_C = null; }

/**
 * Constant for the "Cyrillic Extended-D" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock CYRILLIC_EXTENDED_D;
static { CYRILLIC_EXTENDED_D = null; }

/**
 * Constant for the "Cyrillic Supplement" Unicode character block.
 * This block was previously known as the "Cyrillic Supplementary" block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock CYRILLIC_SUPPLEMENTARY;
static { CYRILLIC_SUPPLEMENTARY = null; }

/**
 * Constant for the "Deseret" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock DESERET;
static { DESERET = null; }

/**
 * Constant for the "Devanagari" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock DEVANAGARI;
static { DEVANAGARI = null; }

/**
 * Constant for the "Devanagari Extended" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock DEVANAGARI_EXTENDED;
static { DEVANAGARI_EXTENDED = null; }

/**
 * Constant for the "Devanagari Extended-A" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock DEVANAGARI_EXTENDED_A;
static { DEVANAGARI_EXTENDED_A = null; }

/**
 * Constant for the "Dingbats" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock DINGBATS;
static { DINGBATS = null; }

/**
 * Constant for the "Dives Akuru" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock DIVES_AKURU;
static { DIVES_AKURU = null; }

/**
 * Constant for the "Dogra" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock DOGRA;
static { DOGRA = null; }

/**
 * Constant for the "Domino Tiles" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock DOMINO_TILES;
static { DOMINO_TILES = null; }

/**
 * Constant for the "Duployan" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock DUPLOYAN;
static { DUPLOYAN = null; }

/**
 * Constant for the "Early Dynastic Cuneiform" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock EARLY_DYNASTIC_CUNEIFORM;
static { EARLY_DYNASTIC_CUNEIFORM = null; }

/**
 * Constant for the "Egyptian Hieroglyphs" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock EGYPTIAN_HIEROGLYPHS;
static { EGYPTIAN_HIEROGLYPHS = null; }

/**
 * Constant for the "Egyptian Hieroglyph Format Controls" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock EGYPTIAN_HIEROGLYPH_FORMAT_CONTROLS;
static { EGYPTIAN_HIEROGLYPH_FORMAT_CONTROLS = null; }

/**
 * Constant for the "Elbasan" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock ELBASAN;
static { ELBASAN = null; }

/**
 * Constant for the "Elymaic" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock ELYMAIC;
static { ELYMAIC = null; }

/**
 * Constant for the "Emoticons" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock EMOTICONS;
static { EMOTICONS = null; }

/**
 * Constant for the "Enclosed Alphanumerics" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ENCLOSED_ALPHANUMERICS;
static { ENCLOSED_ALPHANUMERICS = null; }

/**
 * Constant for the "Enclosed Alphanumeric Supplement" Unicode character
 * block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ENCLOSED_ALPHANUMERIC_SUPPLEMENT;
static { ENCLOSED_ALPHANUMERIC_SUPPLEMENT = null; }

/**
 * Constant for the "Enclosed CJK Letters and Months" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ENCLOSED_CJK_LETTERS_AND_MONTHS;
static { ENCLOSED_CJK_LETTERS_AND_MONTHS = null; }

/**
 * Constant for the "Enclosed Ideographic Supplement" Unicode character
 * block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ENCLOSED_IDEOGRAPHIC_SUPPLEMENT;
static { ENCLOSED_IDEOGRAPHIC_SUPPLEMENT = null; }

/**
 * Constant for the "Ethiopic" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock ETHIOPIC;
static { ETHIOPIC = null; }

/**
 * Constant for the "Ethiopic Extended" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ETHIOPIC_EXTENDED;
static { ETHIOPIC_EXTENDED = null; }

/**
 * Constant for the "Ethiopic Extended-A" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ETHIOPIC_EXTENDED_A;
static { ETHIOPIC_EXTENDED_A = null; }

/**
 * Constant for the "Ethiopic Extended-B" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock ETHIOPIC_EXTENDED_B;
static { ETHIOPIC_EXTENDED_B = null; }

/**
 * Constant for the "Ethiopic Supplement" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock ETHIOPIC_SUPPLEMENT;
static { ETHIOPIC_SUPPLEMENT = null; }

/**
 * Constant for the "General Punctuation" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GENERAL_PUNCTUATION;
static { GENERAL_PUNCTUATION = null; }

/**
 * Constant for the "Geometric Shapes" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GEOMETRIC_SHAPES;
static { GEOMETRIC_SHAPES = null; }

/**
 * Constant for the "Geometric Shapes Extended" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock GEOMETRIC_SHAPES_EXTENDED;
static { GEOMETRIC_SHAPES_EXTENDED = null; }

/**
 * Constant for the "Georgian" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GEORGIAN;
static { GEORGIAN = null; }

/**
 * Constant for the "Georgian Extended" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock GEORGIAN_EXTENDED;
static { GEORGIAN_EXTENDED = null; }

/**
 * Constant for the "Georgian Supplement" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock GEORGIAN_SUPPLEMENT;
static { GEORGIAN_SUPPLEMENT = null; }

/**
 * Constant for the "Glagolitic" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock GLAGOLITIC;
static { GLAGOLITIC = null; }

/**
 * Constant for the "Glagolitic Supplement" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock GLAGOLITIC_SUPPLEMENT;
static { GLAGOLITIC_SUPPLEMENT = null; }

/**
 * Constant for the "Gothic" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock GOTHIC;
static { GOTHIC = null; }

/**
 * Constant for the "Grantha" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock GRANTHA;
static { GRANTHA = null; }

/**
 * Constant for the "Greek and Coptic" Unicode character block.
 * <p>
 * This block was previously known as the "Greek" block.
 *
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GREEK;
static { GREEK = null; }

/**
 * Constant for the "Greek Extended" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GREEK_EXTENDED;
static { GREEK_EXTENDED = null; }

/**
 * Constant for the "Gujarati" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GUJARATI;
static { GUJARATI = null; }

/**
 * Constant for the "Gunjala Gondi" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock GUNJALA_GONDI;
static { GUNJALA_GONDI = null; }

/**
 * Constant for the "Gurmukhi" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock GURMUKHI;
static { GURMUKHI = null; }

/**
 * Constant for the "Halfwidth and Fullwidth Forms" Unicode character
 * block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock HALFWIDTH_AND_FULLWIDTH_FORMS;
static { HALFWIDTH_AND_FULLWIDTH_FORMS = null; }

/**
 * Constant for the "Hangul Compatibility Jamo" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock HANGUL_COMPATIBILITY_JAMO;
static { HANGUL_COMPATIBILITY_JAMO = null; }

/**
 * Constant for the "Hangul Jamo" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock HANGUL_JAMO;
static { HANGUL_JAMO = null; }

/**
 * Constant for the "Hangul Jamo Extended-A" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock HANGUL_JAMO_EXTENDED_A;
static { HANGUL_JAMO_EXTENDED_A = null; }

/**
 * Constant for the "Hangul Jamo Extended-B" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock HANGUL_JAMO_EXTENDED_B;
static { HANGUL_JAMO_EXTENDED_B = null; }

/**
 * Constant for the "Hangul Syllables" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock HANGUL_SYLLABLES;
static { HANGUL_SYLLABLES = null; }

/**
 * Constant for the "Hanifi Rohingya" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock HANIFI_ROHINGYA;
static { HANIFI_ROHINGYA = null; }

/**
 * Constant for the "Hanunoo" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock HANUNOO;
static { HANUNOO = null; }

/**
 * Constant for the "Hatran" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock HATRAN;
static { HATRAN = null; }

/**
 * Constant for the "Hebrew" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock HEBREW;
static { HEBREW = null; }

/**
 * Constant for the "High Private Use Surrogates" Unicode character
 * block.
 * This block represents codepoint values in the private use high
 * surrogate range: U+DB80 through U+DBFF
 *
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock HIGH_PRIVATE_USE_SURROGATES;
static { HIGH_PRIVATE_USE_SURROGATES = null; }

/**
 * Constant for the "High Surrogates" Unicode character block.
 * This block represents codepoint values in the high surrogate
 * range: U+D800 through U+DB7F
 *
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock HIGH_SURROGATES;
static { HIGH_SURROGATES = null; }

/**
 * Constant for the "Hiragana" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock HIRAGANA;
static { HIRAGANA = null; }

/**
 * Constant for the "Ideographic Description Characters" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock IDEOGRAPHIC_DESCRIPTION_CHARACTERS;
static { IDEOGRAPHIC_DESCRIPTION_CHARACTERS = null; }

/**
 * Constant for the "Ideographic Symbols and Punctuation" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock IDEOGRAPHIC_SYMBOLS_AND_PUNCTUATION;
static { IDEOGRAPHIC_SYMBOLS_AND_PUNCTUATION = null; }

/**
 * Constant for the "Imperial Aramaic" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock IMPERIAL_ARAMAIC;
static { IMPERIAL_ARAMAIC = null; }

/**
 * Constant for the "Indic Siyaq Numbers" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock INDIC_SIYAQ_NUMBERS;
static { INDIC_SIYAQ_NUMBERS = null; }

/**
 * Constant for the "Inscriptional Pahlavi" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock INSCRIPTIONAL_PAHLAVI;
static { INSCRIPTIONAL_PAHLAVI = null; }

/**
 * Constant for the "Inscriptional Parthian" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock INSCRIPTIONAL_PARTHIAN;
static { INSCRIPTIONAL_PARTHIAN = null; }

/**
 * Constant for the "IPA Extensions" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock IPA_EXTENSIONS;
static { IPA_EXTENSIONS = null; }

/**
 * Constant for the "Javanese" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock JAVANESE;
static { JAVANESE = null; }

/**
 * Constant for the "Kaithi" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock KAITHI;
static { KAITHI = null; }

/**
 * Constant for the "Kaktovik Numerals" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock KAKTOVIK_NUMERALS;
static { KAKTOVIK_NUMERALS = null; }

/**
 * Constant for the "Kana Extended-A" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock KANA_EXTENDED_A;
static { KANA_EXTENDED_A = null; }

/**
 * Constant for the "Kana Extended-B" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock KANA_EXTENDED_B;
static { KANA_EXTENDED_B = null; }

/**
 * Constant for the "Kana Supplement" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock KANA_SUPPLEMENT;
static { KANA_SUPPLEMENT = null; }

/**
 * Constant for the "Kanbun" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock KANBUN;
static { KANBUN = null; }

/**
 * Constant for the "Kangxi Radicals" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock KANGXI_RADICALS;
static { KANGXI_RADICALS = null; }

/**
 * Constant for the "Kannada" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock KANNADA;
static { KANNADA = null; }

/**
 * Constant for the "Katakana" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock KATAKANA;
static { KATAKANA = null; }

/**
 * Constant for the "Katakana Phonetic Extensions" Unicode character
 * block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock KATAKANA_PHONETIC_EXTENSIONS;
static { KATAKANA_PHONETIC_EXTENSIONS = null; }

/**
 * Constant for the "Kawi" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock KAWI;
static { KAWI = null; }

/**
 * Constant for the "Kayah Li" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock KAYAH_LI;
static { KAYAH_LI = null; }

/**
 * Constant for the "Kharoshthi" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock KHAROSHTHI;
static { KHAROSHTHI = null; }

/**
 * Constant for the "Khitan Small Script" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock KHITAN_SMALL_SCRIPT;
static { KHITAN_SMALL_SCRIPT = null; }

/**
 * Constant for the "Khmer" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock KHMER;
static { KHMER = null; }

/**
 * Constant for the "Khmer Symbols" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock KHMER_SYMBOLS;
static { KHMER_SYMBOLS = null; }

/**
 * Constant for the "Khojki" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock KHOJKI;
static { KHOJKI = null; }

/**
 * Constant for the "Khudawadi" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock KHUDAWADI;
static { KHUDAWADI = null; }

/**
 * Constant for the "Lao" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock LAO;
static { LAO = null; }

/**
 * Constant for the "Latin-1 Supplement" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock LATIN_1_SUPPLEMENT;
static { LATIN_1_SUPPLEMENT = null; }

/**
 * Constant for the "Latin Extended-A" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_A;
static { LATIN_EXTENDED_A = null; }

/**
 * Constant for the "Latin Extended Additional" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_ADDITIONAL;
static { LATIN_EXTENDED_ADDITIONAL = null; }

/**
 * Constant for the "Latin Extended-B" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_B;
static { LATIN_EXTENDED_B = null; }

/**
 * Constant for the "Latin Extended-C" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_C;
static { LATIN_EXTENDED_C = null; }

/**
 * Constant for the "Latin Extended-D" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_D;
static { LATIN_EXTENDED_D = null; }

/**
 * Constant for the "Latin Extended-E" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_E;
static { LATIN_EXTENDED_E = null; }

/**
 * Constant for the "Latin Extended-F" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_F;
static { LATIN_EXTENDED_F = null; }

/**
 * Constant for the "Latin Extended-G" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock LATIN_EXTENDED_G;
static { LATIN_EXTENDED_G = null; }

/**
 * Constant for the "Lepcha" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock LEPCHA;
static { LEPCHA = null; }

/**
 * Constant for the "Letterlike Symbols" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock LETTERLIKE_SYMBOLS;
static { LETTERLIKE_SYMBOLS = null; }

/**
 * Constant for the "Limbu" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock LIMBU;
static { LIMBU = null; }

/**
 * Constant for the "Linear A" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock LINEAR_A;
static { LINEAR_A = null; }

/**
 * Constant for the "Linear B Ideograms" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock LINEAR_B_IDEOGRAMS;
static { LINEAR_B_IDEOGRAMS = null; }

/**
 * Constant for the "Linear B Syllabary" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock LINEAR_B_SYLLABARY;
static { LINEAR_B_SYLLABARY = null; }

/**
 * Constant for the "Lisu" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock LISU;
static { LISU = null; }

/**
 * Constant for the "Lisu Supplement" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock LISU_SUPPLEMENT;
static { LISU_SUPPLEMENT = null; }

/**
 * Constant for the "Low Surrogates" Unicode character block.
 * This block represents codepoint values in the low surrogate
 * range: U+DC00 through U+DFFF
 *
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock LOW_SURROGATES;
static { LOW_SURROGATES = null; }

/**
 * Constant for the "Lycian" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock LYCIAN;
static { LYCIAN = null; }

/**
 * Constant for the "Lydian" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock LYDIAN;
static { LYDIAN = null; }

/**
 * Constant for the "Mahajani" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MAHAJANI;
static { MAHAJANI = null; }

/**
 * Constant for the "Mahjong Tiles" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock MAHJONG_TILES;
static { MAHJONG_TILES = null; }

/**
 * Constant for the "Makasar" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock MAKASAR;
static { MAKASAR = null; }

/**
 * Constant for the "Malayalam" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock MALAYALAM;
static { MALAYALAM = null; }

/**
 * Constant for the "Mandaic" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock MANDAIC;
static { MANDAIC = null; }

/**
 * Constant for the "Manichaean" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MANICHAEAN;
static { MANICHAEAN = null; }

/**
 * Constant for the "Marchen" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock MARCHEN;
static { MARCHEN = null; }

/**
 * Constant for the "Masaram Gondi" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock MASARAM_GONDI;
static { MASARAM_GONDI = null; }

/**
 * Constant for the "Mathematical Alphanumeric Symbols" Unicode
 * character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock MATHEMATICAL_ALPHANUMERIC_SYMBOLS;
static { MATHEMATICAL_ALPHANUMERIC_SYMBOLS = null; }

/**
 * Constant for the "Mathematical Operators" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock MATHEMATICAL_OPERATORS;
static { MATHEMATICAL_OPERATORS = null; }

/**
 * Constant for the "Mayan Numerals" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock MAYAN_NUMERALS;
static { MAYAN_NUMERALS = null; }

/**
 * Constant for the "Medefaidrin" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock MEDEFAIDRIN;
static { MEDEFAIDRIN = null; }

/**
 * Constant for the "Meetei Mayek" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock MEETEI_MAYEK;
static { MEETEI_MAYEK = null; }

/**
 * Constant for the "Meetei Mayek Extensions" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock MEETEI_MAYEK_EXTENSIONS;
static { MEETEI_MAYEK_EXTENSIONS = null; }

/**
 * Constant for the "Mende Kikakui" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MENDE_KIKAKUI;
static { MENDE_KIKAKUI = null; }

/**
 * Constant for the "Meroitic Cursive" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock MEROITIC_CURSIVE;
static { MEROITIC_CURSIVE = null; }

/**
 * Constant for the "Meroitic Hieroglyphs" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock MEROITIC_HIEROGLYPHS;
static { MEROITIC_HIEROGLYPHS = null; }

/**
 * Constant for the "Miao" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock MIAO;
static { MIAO = null; }

/**
 * Constant for the "Miscellaneous Mathematical Symbols-A" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock MISCELLANEOUS_MATHEMATICAL_SYMBOLS_A;
static { MISCELLANEOUS_MATHEMATICAL_SYMBOLS_A = null; }

/**
 * Constant for the "Miscellaneous Mathematical Symbols-B" Unicode
 * character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock MISCELLANEOUS_MATHEMATICAL_SYMBOLS_B;
static { MISCELLANEOUS_MATHEMATICAL_SYMBOLS_B = null; }

/**
 * Constant for the "Miscellaneous Symbols" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock MISCELLANEOUS_SYMBOLS;
static { MISCELLANEOUS_SYMBOLS = null; }

/**
 * Constant for the "Miscellaneous Symbols and Arrows" Unicode character
 * block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock MISCELLANEOUS_SYMBOLS_AND_ARROWS;
static { MISCELLANEOUS_SYMBOLS_AND_ARROWS = null; }

/**
 * Constant for the "Miscellaneous Symbols And Pictographs" Unicode
 * character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock MISCELLANEOUS_SYMBOLS_AND_PICTOGRAPHS;
static { MISCELLANEOUS_SYMBOLS_AND_PICTOGRAPHS = null; }

/**
 * Constant for the "Miscellaneous Technical" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock MISCELLANEOUS_TECHNICAL;
static { MISCELLANEOUS_TECHNICAL = null; }

/**
 * Constant for the "Modi" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MODI;
static { MODI = null; }

/**
 * Constant for the "Modifier Tone Letters" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock MODIFIER_TONE_LETTERS;
static { MODIFIER_TONE_LETTERS = null; }

/**
 * Constant for the "Mongolian" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock MONGOLIAN;
static { MONGOLIAN = null; }

/**
 * Constant for the "Mongolian Supplement" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock MONGOLIAN_SUPPLEMENT;
static { MONGOLIAN_SUPPLEMENT = null; }

/**
 * Constant for the "Mro" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MRO;
static { MRO = null; }

/**
 * Constant for the "Multani" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MULTANI;
static { MULTANI = null; }

/**
 * Constant for the "Musical Symbols" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock MUSICAL_SYMBOLS;
static { MUSICAL_SYMBOLS = null; }

/**
 * Constant for the "Myanmar" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock MYANMAR;
static { MYANMAR = null; }

/**
 * Constant for the "Myanmar Extended-A" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock MYANMAR_EXTENDED_A;
static { MYANMAR_EXTENDED_A = null; }

/**
 * Constant for the "Myanmar Extended-B" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock MYANMAR_EXTENDED_B;
static { MYANMAR_EXTENDED_B = null; }

/**
 * Constant for the "Nabataean" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock NABATAEAN;
static { NABATAEAN = null; }

/**
 * Constant for the "Nag Mundari" Unicode
 * character block.
 * @since 20
 */

public static final java.lang.Character.UnicodeBlock NAG_MUNDARI;
static { NAG_MUNDARI = null; }

/**
 * Constant for the "Nandinagari" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock NANDINAGARI;
static { NANDINAGARI = null; }

/**
 * Constant for the "Newa" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock NEWA;
static { NEWA = null; }

/**
 * Constant for the "New Tai Lue" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock NEW_TAI_LUE;
static { NEW_TAI_LUE = null; }

/**
 * Constant for the "NKo" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock NKO;
static { NKO = null; }

/**
 * Constant for the "Number Forms" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock NUMBER_FORMS;
static { NUMBER_FORMS = null; }

/**
 * Constant for the "Nushu" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock NUSHU;
static { NUSHU = null; }

/**
 * Constant for the "Nyiakeng Puachue Hmong" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock NYIAKENG_PUACHUE_HMONG;
static { NYIAKENG_PUACHUE_HMONG = null; }

/**
 * Constant for the "Ogham" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock OGHAM;
static { OGHAM = null; }

/**
 * Constant for the "Old Hungarian" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock OLD_HUNGARIAN;
static { OLD_HUNGARIAN = null; }

/**
 * Constant for the "Old Italic" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock OLD_ITALIC;
static { OLD_ITALIC = null; }

/**
 * Constant for the "Old North Arabian" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock OLD_NORTH_ARABIAN;
static { OLD_NORTH_ARABIAN = null; }

/**
 * Constant for the "Old Permic" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock OLD_PERMIC;
static { OLD_PERMIC = null; }

/**
 * Constant for the "Old Persian" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock OLD_PERSIAN;
static { OLD_PERSIAN = null; }

/**
 * Constant for the "Old Sogdian" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock OLD_SOGDIAN;
static { OLD_SOGDIAN = null; }

/**
 * Constant for the "Old South Arabian" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock OLD_SOUTH_ARABIAN;
static { OLD_SOUTH_ARABIAN = null; }

/**
 * Constant for the "Old Turkic" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock OLD_TURKIC;
static { OLD_TURKIC = null; }

/**
 * Constant for the "Old Uyghur" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock OLD_UYGHUR;
static { OLD_UYGHUR = null; }

/**
 * Constant for the "Ol Chiki" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock OL_CHIKI;
static { OL_CHIKI = null; }

/**
 * Constant for the "Optical Character Recognition" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock OPTICAL_CHARACTER_RECOGNITION;
static { OPTICAL_CHARACTER_RECOGNITION = null; }

/**
 * Constant for the "Oriya" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock ORIYA;
static { ORIYA = null; }

/**
 * Constant for the "Ornamental Dingbats" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock ORNAMENTAL_DINGBATS;
static { ORNAMENTAL_DINGBATS = null; }

/**
 * Constant for the "Osage" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock OSAGE;
static { OSAGE = null; }

/**
 * Constant for the "Osmanya" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock OSMANYA;
static { OSMANYA = null; }

/**
 * Constant for the "Ottoman Siyaq Numbers" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock OTTOMAN_SIYAQ_NUMBERS;
static { OTTOMAN_SIYAQ_NUMBERS = null; }

/**
 * Constant for the "Pahawh Hmong" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock PAHAWH_HMONG;
static { PAHAWH_HMONG = null; }

/**
 * Constant for the "Palmyrene" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock PALMYRENE;
static { PALMYRENE = null; }

/**
 * Constant for the "Pau Cin Hau" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock PAU_CIN_HAU;
static { PAU_CIN_HAU = null; }

/**
 * Constant for the "Phags-pa" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock PHAGS_PA;
static { PHAGS_PA = null; }

/**
 * Constant for the "Phaistos Disc" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock PHAISTOS_DISC;
static { PHAISTOS_DISC = null; }

/**
 * Constant for the "Phoenician" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock PHOENICIAN;
static { PHOENICIAN = null; }

/**
 * Constant for the "Phonetic Extensions" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock PHONETIC_EXTENSIONS;
static { PHONETIC_EXTENSIONS = null; }

/**
 * Constant for the "Phonetic Extensions Supplement" Unicode character
 * block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock PHONETIC_EXTENSIONS_SUPPLEMENT;
static { PHONETIC_EXTENSIONS_SUPPLEMENT = null; }

/**
 * Constant for the "Playing Cards" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock PLAYING_CARDS;
static { PLAYING_CARDS = null; }

/**
 * Constant for the "Private Use Area" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock PRIVATE_USE_AREA;
static { PRIVATE_USE_AREA = null; }

/**
 * Constant for the "Psalter Pahlavi" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock PSALTER_PAHLAVI;
static { PSALTER_PAHLAVI = null; }

/**
 * Constant for the "Rejang" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock REJANG;
static { REJANG = null; }

/**
 * Constant for the "Rumi Numeral Symbols" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock RUMI_NUMERAL_SYMBOLS;
static { RUMI_NUMERAL_SYMBOLS = null; }

/**
 * Constant for the "Runic" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock RUNIC;
static { RUNIC = null; }

/**
 * Constant for the "Samaritan" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock SAMARITAN;
static { SAMARITAN = null; }

/**
 * Constant for the "Saurashtra" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock SAURASHTRA;
static { SAURASHTRA = null; }

/**
 * Constant for the "Sharada" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock SHARADA;
static { SHARADA = null; }

/**
 * Constant for the "Shavian" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock SHAVIAN;
static { SHAVIAN = null; }

/**
 * Constant for the "Shorthand Format Controls" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock SHORTHAND_FORMAT_CONTROLS;
static { SHORTHAND_FORMAT_CONTROLS = null; }

/**
 * Constant for the "Siddham" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock SIDDHAM;
static { SIDDHAM = null; }

/**
 * Constant for the "Sinhala" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock SINHALA;
static { SINHALA = null; }

/**
 * Constant for the "Sinhala Archaic Numbers" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock SINHALA_ARCHAIC_NUMBERS;
static { SINHALA_ARCHAIC_NUMBERS = null; }

/**
 * Constant for the "Small Form Variants" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock SMALL_FORM_VARIANTS;
static { SMALL_FORM_VARIANTS = null; }

/**
 * Constant for the "Small Kana Extension" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock SMALL_KANA_EXTENSION;
static { SMALL_KANA_EXTENSION = null; }

/**
 * Constant for the "Sogdian" Unicode
 * character block.
 * @since 12
 */

public static final java.lang.Character.UnicodeBlock SOGDIAN;
static { SOGDIAN = null; }

/**
 * Constant for the "Sora Sompeng" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock SORA_SOMPENG;
static { SORA_SOMPENG = null; }

/**
 * Constant for the "Soyombo" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock SOYOMBO;
static { SOYOMBO = null; }

/**
 * Constant for the "Spacing Modifier Letters" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock SPACING_MODIFIER_LETTERS;
static { SPACING_MODIFIER_LETTERS = null; }

/**
 * Constant for the "Specials" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock SPECIALS;
static { SPECIALS = null; }

/**
 * Constant for the "Sundanese" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock SUNDANESE;
static { SUNDANESE = null; }

/**
 * Constant for the "Sundanese Supplement" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock SUNDANESE_SUPPLEMENT;
static { SUNDANESE_SUPPLEMENT = null; }

/**
 * Constant for the "Superscripts and Subscripts" Unicode character
 * block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock SUPERSCRIPTS_AND_SUBSCRIPTS;
static { SUPERSCRIPTS_AND_SUBSCRIPTS = null; }

/**
 * Constant for the "Supplemental Arrows-A" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTAL_ARROWS_A;
static { SUPPLEMENTAL_ARROWS_A = null; }

/**
 * Constant for the "Supplemental Arrows-B" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTAL_ARROWS_B;
static { SUPPLEMENTAL_ARROWS_B = null; }

/**
 * Constant for the "Supplemental Arrows-C" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTAL_ARROWS_C;
static { SUPPLEMENTAL_ARROWS_C = null; }

/**
 * Constant for the "Supplemental Mathematical Operators" Unicode
 * character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTAL_MATHEMATICAL_OPERATORS;
static { SUPPLEMENTAL_MATHEMATICAL_OPERATORS = null; }

/**
 * Constant for the "Supplemental Punctuation" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTAL_PUNCTUATION;
static { SUPPLEMENTAL_PUNCTUATION = null; }

/**
 * Constant for the "Supplemental Symbols and Pictographs" Unicode
 * character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTAL_SYMBOLS_AND_PICTOGRAPHS;
static { SUPPLEMENTAL_SYMBOLS_AND_PICTOGRAPHS = null; }

/**
 * Constant for the "Supplementary Private Use Area-A" Unicode character
 * block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTARY_PRIVATE_USE_AREA_A;
static { SUPPLEMENTARY_PRIVATE_USE_AREA_A = null; }

/**
 * Constant for the "Supplementary Private Use Area-B" Unicode character
 * block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock SUPPLEMENTARY_PRIVATE_USE_AREA_B;
static { SUPPLEMENTARY_PRIVATE_USE_AREA_B = null; }

/**
 * @deprecated
 * Instead of {@code SURROGATES_AREA}, use {@link #HIGH_SURROGATES},
 * {@link #HIGH_PRIVATE_USE_SURROGATES}, and {@link #LOW_SURROGATES}.
 * These constants match the block definitions of the Unicode Standard.
 * The {@link #of(char)} and {@link #of(int)} methods return the
 * standard constants.
 */

@Deprecated public static final java.lang.Character.UnicodeBlock SURROGATES_AREA;
static { SURROGATES_AREA = null; }

/**
 * Constant for the "Sutton SignWriting" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock SUTTON_SIGNWRITING;
static { SUTTON_SIGNWRITING = null; }

/**
 * Constant for the "Syloti Nagri" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock SYLOTI_NAGRI;
static { SYLOTI_NAGRI = null; }

/**
 * Constant for the "Symbols and Pictographs Extended-A" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock SYMBOLS_AND_PICTOGRAPHS_EXTENDED_A;
static { SYMBOLS_AND_PICTOGRAPHS_EXTENDED_A = null; }

/**
 * Constant for the "Symbols for Legacy Computing" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock SYMBOLS_FOR_LEGACY_COMPUTING;
static { SYMBOLS_FOR_LEGACY_COMPUTING = null; }

/**
 * Constant for the "Syriac" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock SYRIAC;
static { SYRIAC = null; }

/**
 * Constant for the "Syriac Supplement" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock SYRIAC_SUPPLEMENT;
static { SYRIAC_SUPPLEMENT = null; }

/**
 * Constant for the "Tagalog" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock TAGALOG;
static { TAGALOG = null; }

/**
 * Constant for the "Tagbanwa" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock TAGBANWA;
static { TAGBANWA = null; }

/**
 * Constant for the "Tags" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock TAGS;
static { TAGS = null; }

/**
 * Constant for the "Tai Le" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock TAI_LE;
static { TAI_LE = null; }

/**
 * Constant for the "Tai Tham" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock TAI_THAM;
static { TAI_THAM = null; }

/**
 * Constant for the "Tai Viet" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock TAI_VIET;
static { TAI_VIET = null; }

/**
 * Constant for the "Tai Xuan Jing Symbols" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock TAI_XUAN_JING_SYMBOLS;
static { TAI_XUAN_JING_SYMBOLS = null; }

/**
 * Constant for the "Takri" Unicode character block.
 * @since 1.8
 */

public static final java.lang.Character.UnicodeBlock TAKRI;
static { TAKRI = null; }

/**
 * Constant for the "Tamil" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock TAMIL;
static { TAMIL = null; }

/**
 * Constant for the "Tamil Supplement" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock TAMIL_SUPPLEMENT;
static { TAMIL_SUPPLEMENT = null; }

/**
 * Constant for the "Tangsa" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock TANGSA;
static { TANGSA = null; }

/**
 * Constant for the "Tangut" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock TANGUT;
static { TANGUT = null; }

/**
 * Constant for the "Tangut Components" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock TANGUT_COMPONENTS;
static { TANGUT_COMPONENTS = null; }

/**
 * Constant for the "Tangut Supplement" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock TANGUT_SUPPLEMENT;
static { TANGUT_SUPPLEMENT = null; }

/**
 * Constant for the "Telugu" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock TELUGU;
static { TELUGU = null; }

/**
 * Constant for the "Thaana" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock THAANA;
static { THAANA = null; }

/**
 * Constant for the "Thai" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock THAI;
static { THAI = null; }

/**
 * Constant for the "Tibetan" Unicode character block.
 * @since 1.2
 */

public static final java.lang.Character.UnicodeBlock TIBETAN;
static { TIBETAN = null; }

/**
 * Constant for the "Tifinagh" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock TIFINAGH;
static { TIFINAGH = null; }

/**
 * Constant for the "Tirhuta" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock TIRHUTA;
static { TIRHUTA = null; }

/**
 * Constant for the "Toto" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock TOTO;
static { TOTO = null; }

/**
 * Constant for the "Transport And Map Symbols" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock TRANSPORT_AND_MAP_SYMBOLS;
static { TRANSPORT_AND_MAP_SYMBOLS = null; }

/**
 * Constant for the "Ugaritic" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock UGARITIC;
static { UGARITIC = null; }

/**
 * Constant for the "Unified Canadian Aboriginal Syllabics" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS;
static { UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS = null; }

/**
 * Constant for the "Unified Canadian Aboriginal Syllabics Extended"
 * Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS_EXTENDED;
static { UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS_EXTENDED = null; }

/**
 * Constant for the "Unified Canadian Aboriginal Syllabics Extended-A" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS_EXTENDED_A;
static { UNIFIED_CANADIAN_ABORIGINAL_SYLLABICS_EXTENDED_A = null; }

/**
 * Constant for the "Vai" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock VAI;
static { VAI = null; }

/**
 * Constant for the "Variation Selectors" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock VARIATION_SELECTORS;
static { VARIATION_SELECTORS = null; }

/**
 * Constant for the "Variation Selectors Supplement" Unicode character
 * block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock VARIATION_SELECTORS_SUPPLEMENT;
static { VARIATION_SELECTORS_SUPPLEMENT = null; }

/**
 * Constant for the "Vedic Extensions" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock VEDIC_EXTENSIONS;
static { VEDIC_EXTENSIONS = null; }

/**
 * Constant for the "Vertical Forms" Unicode character block.
 * @since 1.7
 */

public static final java.lang.Character.UnicodeBlock VERTICAL_FORMS;
static { VERTICAL_FORMS = null; }

/**
 * Constant for the "Vithkuqi" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock VITHKUQI;
static { VITHKUQI = null; }

/**
 * Constant for the "Wancho" Unicode
 * character block.
 * @since 13
 */

public static final java.lang.Character.UnicodeBlock WANCHO;
static { WANCHO = null; }

/**
 * Constant for the "Warang Citi" Unicode character block.
 * @since 9
 */

public static final java.lang.Character.UnicodeBlock WARANG_CITI;
static { WARANG_CITI = null; }

/**
 * Constant for the "Yezidi" Unicode
 * character block.
 * @since 15
 */

public static final java.lang.Character.UnicodeBlock YEZIDI;
static { YEZIDI = null; }

/**
 * Constant for the "Yijing Hexagram Symbols" Unicode character block.
 * @since 1.5
 */

public static final java.lang.Character.UnicodeBlock YIJING_HEXAGRAM_SYMBOLS;
static { YIJING_HEXAGRAM_SYMBOLS = null; }

/**
 * Constant for the "Yi Radicals" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock YI_RADICALS;
static { YI_RADICALS = null; }

/**
 * Constant for the "Yi Syllables" Unicode character block.
 * @since 1.4
 */

public static final java.lang.Character.UnicodeBlock YI_SYLLABLES;
static { YI_SYLLABLES = null; }

/**
 * Constant for the "Zanabazar Square" Unicode
 * character block.
 * @since 11
 */

public static final java.lang.Character.UnicodeBlock ZANABAZAR_SQUARE;
static { ZANABAZAR_SQUARE = null; }

/**
 * Constant for the "Znamenny Musical Notation" Unicode
 * character block.
 * @since 19
 */

public static final java.lang.Character.UnicodeBlock ZNAMENNY_MUSICAL_NOTATION;
static { ZNAMENNY_MUSICAL_NOTATION = null; }
}

/**
 * A family of character subsets representing the character scripts
 * defined in the <a href="http://www.unicode.org/reports/tr24/">
 * <i>Unicode Standard Annex #24: Script Names</i></a>. Every Unicode
 * character is assigned to a single Unicode script, either a specific
 * script, such as {@link java.lang.Character.UnicodeScript#LATIN Latin}, or
 * one of the following three special values,
 * {@link java.lang.Character.UnicodeScript#INHERITED Inherited},
 * {@link java.lang.Character.UnicodeScript#COMMON Common} or
 * {@link java.lang.Character.UnicodeScript#UNKNOWN Unknown}.
 *
 * @spec https://www.unicode.org/reports/tr24 Unicode Script Property
 * @since 1.7
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public enum UnicodeScript {
/**
 * Unicode script "Adlam".
 * @since 11
 */

ADLAM,
/**
 * Unicode script "Ahom".
 * @since 9
 */

AHOM,
/**
 * Unicode script "Anatolian Hieroglyphs".
 * @since 9
 */

ANATOLIAN_HIEROGLYPHS,
/**
 * Unicode script "Arabic".
 */

ARABIC,
/**
 * Unicode script "Armenian".
 */

ARMENIAN,
/**
 * Unicode script "Avestan".
 */

AVESTAN,
/**
 * Unicode script "Balinese".
 */

BALINESE,
/**
 * Unicode script "Bamum".
 */

BAMUM,
/**
 * Unicode script "Bassa Vah".
 * @since 9
 */

BASSA_VAH,
/**
 * Unicode script "Batak".
 */

BATAK,
/**
 * Unicode script "Bengali".
 */

BENGALI,
/**
 * Unicode script "Bhaiksuki".
 * @since 11
 */

BHAIKSUKI,
/**
 * Unicode script "Bopomofo".
 */

BOPOMOFO,
/**
 * Unicode script "Brahmi".
 */

BRAHMI,
/**
 * Unicode script "Braille".
 */

BRAILLE,
/**
 * Unicode script "Buginese".
 */

BUGINESE,
/**
 * Unicode script "Buhid".
 */

BUHID,
/**
 * Unicode script "Canadian_Aboriginal".
 */

CANADIAN_ABORIGINAL,
/**
 * Unicode script "Carian".
 */

CARIAN,
/**
 * Unicode script "Caucasian Albanian".
 * @since 9
 */

CAUCASIAN_ALBANIAN,
/**
 * Unicode script "Chakma".
 * @since 1.8
 */

CHAKMA,
/**
 * Unicode script "Cham".
 */

CHAM,
/**
 * Unicode script "Cherokee".
 */

CHEROKEE,
/**
 * Unicode script "Chorasmian".
 * @since 15
 */

CHORASMIAN,
/**
 * Unicode script "Common".
 */

COMMON,
/**
 * Unicode script "Coptic".
 */

COPTIC,
/**
 * Unicode script "Cuneiform".
 */

CUNEIFORM,
/**
 * Unicode script "Cypriot".
 */

CYPRIOT,
/**
 * Unicode script "Cypro Minoan".
 * @since 19
 */

CYPRO_MINOAN,
/**
 * Unicode script "Cyrillic".
 */

CYRILLIC,
/**
 * Unicode script "Deseret".
 */

DESERET,
/**
 * Unicode script "Devanagari".
 */

DEVANAGARI,
/**
 * Unicode script "Dives Akuru".
 * @since 15
 */

DIVES_AKURU,
/**
 * Unicode script "Dogra".
 * @since 12
 */

DOGRA,
/**
 * Unicode script "Duployan".
 * @since 9
 */

DUPLOYAN,
/**
 * Unicode script "Egyptian_Hieroglyphs".
 */

EGYPTIAN_HIEROGLYPHS,
/**
 * Unicode script "Elbasan".
 * @since 9
 */

ELBASAN,
/**
 * Unicode script "Elymaic".
 * @since 13
 */

ELYMAIC,
/**
 * Unicode script "Ethiopic".
 */

ETHIOPIC,
/**
 * Unicode script "Georgian".
 */

GEORGIAN,
/**
 * Unicode script "Glagolitic".
 */

GLAGOLITIC,
/**
 * Unicode script "Gothic".
 */

GOTHIC,
/**
 * Unicode script "Grantha".
 * @since 9
 */

GRANTHA,
/**
 * Unicode script "Greek".
 */

GREEK,
/**
 * Unicode script "Gujarati".
 */

GUJARATI,
/**
 * Unicode script "Gunjala Gondi".
 * @since 12
 */

GUNJALA_GONDI,
/**
 * Unicode script "Gurmukhi".
 */

GURMUKHI,
/**
 * Unicode script "Han".
 */

HAN,
/**
 * Unicode script "Hangul".
 */

HANGUL,
/**
 * Unicode script "Hanifi Rohingya".
 * @since 12
 */

HANIFI_ROHINGYA,
/**
 * Unicode script "Hanunoo".
 */

HANUNOO,
/**
 * Unicode script "Hatran".
 * @since 9
 */

HATRAN,
/**
 * Unicode script "Hebrew".
 */

HEBREW,
/**
 * Unicode script "Hiragana".
 */

HIRAGANA,
/**
 * Unicode script "Imperial_Aramaic".
 */

IMPERIAL_ARAMAIC,
/**
 * Unicode script "Inherited".
 */

INHERITED,
/**
 * Unicode script "Inscriptional_Pahlavi".
 */

INSCRIPTIONAL_PAHLAVI,
/**
 * Unicode script "Inscriptional_Parthian".
 */

INSCRIPTIONAL_PARTHIAN,
/**
 * Unicode script "Javanese".
 */

JAVANESE,
/**
 * Unicode script "Kaithi".
 */

KAITHI,
/**
 * Unicode script "Kannada".
 */

KANNADA,
/**
 * Unicode script "Katakana".
 */

KATAKANA,
/**
 * Unicode script "Kawi".
 * @since 20
 */

KAWI,
/**
 * Unicode script "Kayah_Li".
 */

KAYAH_LI,
/**
 * Unicode script "Kharoshthi".
 */

KHAROSHTHI,
/**
 * Unicode script "Khitan Small Script".
 * @since 15
 */

KHITAN_SMALL_SCRIPT,
/**
 * Unicode script "Khmer".
 */

KHMER,
/**
 * Unicode script "Khojki".
 * @since 9
 */

KHOJKI,
/**
 * Unicode script "Khudawadi".
 * @since 9
 */

KHUDAWADI,
/**
 * Unicode script "Lao".
 */

LAO,
/**
 * Unicode script "Latin".
 */

LATIN,
/**
 * Unicode script "Lepcha".
 */

LEPCHA,
/**
 * Unicode script "Limbu".
 */

LIMBU,
/**
 * Unicode script "Linear A".
 * @since 9
 */

LINEAR_A,
/**
 * Unicode script "Linear_B".
 */

LINEAR_B,
/**
 * Unicode script "Lisu".
 */

LISU,
/**
 * Unicode script "Lycian".
 */

LYCIAN,
/**
 * Unicode script "Lydian".
 */

LYDIAN,
/**
 * Unicode script "Mahajani".
 * @since 9
 */

MAHAJANI,
/**
 * Unicode script "Makasar".
 * @since 12
 */

MAKASAR,
/**
 * Unicode script "Malayalam".
 */

MALAYALAM,
/**
 * Unicode script "Mandaic".
 */

MANDAIC,
/**
 * Unicode script "Manichaean".
 * @since 9
 */

MANICHAEAN,
/**
 * Unicode script "Marchen".
 * @since 11
 */

MARCHEN,
/**
 * Unicode script "Masaram Gondi".
 * @since 11
 */

MASARAM_GONDI,
/**
 * Unicode script "Medefaidrin".
 * @since 12
 */

MEDEFAIDRIN,
/**
 * Unicode script "Meetei_Mayek".
 */

MEETEI_MAYEK,
/**
 * Unicode script "Mende Kikakui".
 * @since 9
 */

MENDE_KIKAKUI,
/**
 * Unicode script "Meroitic Cursive".
 * @since 1.8
 */

MEROITIC_CURSIVE,
/**
 * Unicode script "Meroitic Hieroglyphs".
 * @since 1.8
 */

MEROITIC_HIEROGLYPHS,
/**
 * Unicode script "Miao".
 * @since 1.8
 */

MIAO,
/**
 * Unicode script "Modi".
 * @since 9
 */

MODI,
/**
 * Unicode script "Mongolian".
 */

MONGOLIAN,
/**
 * Unicode script "Mro".
 * @since 9
 */

MRO,
/**
 * Unicode script "Multani".
 * @since 9
 */

MULTANI,
/**
 * Unicode script "Myanmar".
 */

MYANMAR,
/**
 * Unicode script "Nabataean".
 * @since 9
 */

NABATAEAN,
/**
 * Unicode script "Nag Mundari".
 * @since 20
 */

NAG_MUNDARI,
/**
 * Unicode script "Nandinagari".
 * @since 13
 */

NANDINAGARI,
/**
 * Unicode script "Newa".
 * @since 11
 */

NEWA,
/**
 * Unicode script "New_Tai_Lue".
 */

NEW_TAI_LUE,
/**
 * Unicode script "Nko".
 */

NKO,
/**
 * Unicode script "Nushu".
 * @since 11
 */

NUSHU,
/**
 * Unicode script "Nyiakeng Puachue Hmong".
 * @since 13
 */

NYIAKENG_PUACHUE_HMONG,
/**
 * Unicode script "Ogham".
 */

OGHAM,
/**
 * Unicode script "Old Hungarian".
 * @since 9
 */

OLD_HUNGARIAN,
/**
 * Unicode script "Old_Italic".
 */

OLD_ITALIC,
/**
 * Unicode script "Old North Arabian".
 * @since 9
 */

OLD_NORTH_ARABIAN,
/**
 * Unicode script "Old Permic".
 * @since 9
 */

OLD_PERMIC,
/**
 * Unicode script "Old_Persian".
 */

OLD_PERSIAN,
/**
 * Unicode script "Old Sogdian".
 * @since 12
 */

OLD_SOGDIAN,
/**
 * Unicode script "Old_South_Arabian".
 */

OLD_SOUTH_ARABIAN,
/**
 * Unicode script "Old_Turkic".
 */

OLD_TURKIC,
/**
 * Unicode script "Old Uyghur".
 * @since 19
 */

OLD_UYGHUR,
/**
 * Unicode script "Ol_Chiki".
 */

OL_CHIKI,
/**
 * Unicode script "Oriya".
 */

ORIYA,
/**
 * Unicode script "Osage".
 * @since 11
 */

OSAGE,
/**
 * Unicode script "Osmanya".
 */

OSMANYA,
/**
 * Unicode script "Pahawh Hmong".
 * @since 9
 */

PAHAWH_HMONG,
/**
 * Unicode script "Palmyrene".
 * @since 9
 */

PALMYRENE,
/**
 * Unicode script "Pau Cin Hau".
 * @since 9
 */

PAU_CIN_HAU,
/**
 * Unicode script "Phags_Pa".
 */

PHAGS_PA,
/**
 * Unicode script "Phoenician".
 */

PHOENICIAN,
/**
 * Unicode script "Psalter Pahlavi".
 * @since 9
 */

PSALTER_PAHLAVI,
/**
 * Unicode script "Rejang".
 */

REJANG,
/**
 * Unicode script "Runic".
 */

RUNIC,
/**
 * Unicode script "Samaritan".
 */

SAMARITAN,
/**
 * Unicode script "Saurashtra".
 */

SAURASHTRA,
/**
 * Unicode script "Sharada".
 * @since 1.8
 */

SHARADA,
/**
 * Unicode script "Shavian".
 */

SHAVIAN,
/**
 * Unicode script "Siddham".
 * @since 9
 */

SIDDHAM,
/**
 * Unicode script "SignWriting".
 * @since 9
 */

SIGNWRITING,
/**
 * Unicode script "Sinhala".
 */

SINHALA,
/**
 * Unicode script "Sogdian".
 * @since 12
 */

SOGDIAN,
/**
 * Unicode script "Sora Sompeng".
 * @since 1.8
 */

SORA_SOMPENG,
/**
 * Unicode script "Soyombo".
 * @since 11
 */

SOYOMBO,
/**
 * Unicode script "Sundanese".
 */

SUNDANESE,
/**
 * Unicode script "Syloti_Nagri".
 */

SYLOTI_NAGRI,
/**
 * Unicode script "Syriac".
 */

SYRIAC,
/**
 * Unicode script "Tagalog".
 */

TAGALOG,
/**
 * Unicode script "Tagbanwa".
 */

TAGBANWA,
/**
 * Unicode script "Tai_Le".
 */

TAI_LE,
/**
 * Unicode script "Tai_Tham".
 */

TAI_THAM,
/**
 * Unicode script "Tai_Viet".
 */

TAI_VIET,
/**
 * Unicode script "Takri".
 * @since 1.8
 */

TAKRI,
/**
 * Unicode script "Tamil".
 */

TAMIL,
/**
 * Unicode script "Tangsa".
 * @since 19
 */

TANGSA,
/**
 * Unicode script "Tangut".
 * @since 11
 */

TANGUT,
/**
 * Unicode script "Telugu".
 */

TELUGU,
/**
 * Unicode script "Thaana".
 */

THAANA,
/**
 * Unicode script "Thai".
 */

THAI,
/**
 * Unicode script "Tibetan".
 */

TIBETAN,
/**
 * Unicode script "Tifinagh".
 */

TIFINAGH,
/**
 * Unicode script "Tirhuta".
 * @since 9
 */

TIRHUTA,
/**
 * Unicode script "Toto".
 * @since 19
 */

TOTO,
/**
 * Unicode script "Ugaritic".
 */

UGARITIC,
/**
 * Unicode script "Unknown".
 */

UNKNOWN,
/**
 * Unicode script "Vai".
 */

VAI,
/**
 * Unicode script "Vithkuqi".
 * @since 19
 */

VITHKUQI,
/**
 * Unicode script "Wancho".
 * @since 13
 */

WANCHO,
/**
 * Unicode script "Warang Citi".
 * @since 9
 */

WARANG_CITI,
/**
 * Unicode script "Yezidi".
 * @since 15
 */

YEZIDI,
/**
 * Unicode script "Yi".
 */

YI,
/**
 * Unicode script "Zanabazar Square".
 * @since 11
 */

ZANABAZAR_SQUARE;

/**
 * Returns the UnicodeScript constant with the given Unicode script
 * name or the script name alias. Script names and their aliases are
 * determined by The Unicode Standard. The files {@code Scripts.txt}
 * and {@code PropertyValueAliases.txt} define script names
 * and the script name aliases for a particular version of the
 * standard. The {@link java.lang.Character Character} class specifies the version of
 * the standard that it supports.
 * <p>
 * Character case is ignored for all of the valid script names.
 * The en_US locale's case mapping rules are used to provide
 * case-insensitive string comparisons for script name validation.
 *
 * @param scriptName A {@code UnicodeScript} name.
 * @return The {@code UnicodeScript} constant identified
 *         by {@code scriptName}
 * @throws java.lang.IllegalArgumentException if {@code scriptName} is an
 *         invalid name
 * @throws java.lang.NullPointerException if {@code scriptName} is null
 */

@androidx.annotation.RecentlyNonNull
public static java.lang.Character.UnicodeScript forName(@androidx.annotation.RecentlyNonNull java.lang.String scriptName) { throw new RuntimeException("Stub!"); }

/**
 * Returns the enum constant representing the Unicode script of which
 * the given character (Unicode code point) is assigned to.
 *
 * @param   codePoint the character (Unicode code point) in question.
 * @return  The {@code UnicodeScript} constant representing the
 *          Unicode script of which this character is assigned to.
 *
 * @throws  java.lang.IllegalArgumentException if the specified
 * {@code codePoint} is an invalid Unicode code point.
 * @see java.lang.Character#isValidCodePoint(int)
 *
 */

@androidx.annotation.RecentlyNonNull
public static java.lang.Character.UnicodeScript of(int codePoint) { throw new RuntimeException("Stub!"); }
}

}

