/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.ranging.oob;

import java.util.concurrent.Executor;

/**
 * TransportHandle is used as the Out-Of-Band (OOB) transport mechanism by ranging module.
 * In cases where module is used in a non-raw ranging mode, the user shall provide an implementation
 * of the TransportHandle, allowing ranging module to do the necessary OOB communication with a peer
 * device using the provided transport handle. Some examples of OOB transport between two peer
 * devices are:
 * <ul>
 *     <li>BLE GATT connection</li>
 *     <li>Wi-Fi MDNS link</li>
 *     <li>Internet</li>
 * </ul>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface TransportHandle extends java.lang.AutoCloseable {

/**
 * Registers a callback to receive updates from the transport mechanism.
 *
 * <p>The callback should be used to notify information about the peer device including the
 * data received from the peer device.
 *
 * @param executor the {@link java.util.concurrent.Executor Executor} on which the callback should be invoked. Must not be
 *                 null.
 * @param callback the {@link android.ranging.oob.TransportHandle.ReceiveCallback ReceiveCallback} instance to receive updates. Must not be null.
 * @throws java.lang.IllegalArgumentException if either {@code executor} or {@code callback} is null.
 */

public void registerReceiveCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.ranging.oob.TransportHandle.ReceiveCallback callback);

/** Send data to the peer device via the implemented OOB transport.
 *
 * @param data the data to be sent to the peer device. Must not be null.
 * @throws java.lang.IllegalArgumentException if the provided data is null or invalid
 */

public void sendData(@android.annotation.NonNull byte[] data);
/** TransportHandle callback. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface ReceiveCallback {

/**
 * Notifies the receiver that the TransportHandle instance can't be used anymore to receive
 * or send data. Also call this in {@link java.lang.AutoCloseable#close() AutoCloseable.close()}.
 */

public void onClose();

/**
 * Notifies the receiver that the TransportHandle instance can't be used to receive or send
 * data until {@see onReconnect()} is called.
 */

public void onDisconnect();

/**
 * Notifies and provides data received from the peer device.
 *
 * @param data the data received from the peer device. Must not be null.
 */

public void onReceiveData(@android.annotation.NonNull byte[] data);

/**
 * Notifies the receiver the TransportHandle instance can be used again to send and receive
 * data. Should only be called if {@see onDisconnect()} preceded it.
 */

public void onReconnect();

/**
 * Called when a data send operation fails.
 */

public void onSendFailed();
}

}

